#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTest.VAL;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
#endregion

namespace gov.va.med.vbecs.UnitTests
{
	/// <summary>
	/// Code for creating unit test data
	/// </summary>
	public class DataCreator
	{

		static DataTable _DT_BloodProducts = null;
		static DataTable _DT_ProductsWithTargets = null;
		//static DataTable _DT_ProductsWithIrradiatedTargets = null;


		public static void CreateWorkloadProcess(Common.WorkloadProcessID processId)
		{
			//Create WorkloadProcess records
			Guid randomProcessGuid =  DataCreator.GetRandomRecordGuid(TABLES.VbecsProcess.TableName, "");
			if (processId != Common.WorkloadProcessID.Unknown)
			{
				DataTable dataCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM WorkloadProcess WHERE VbecsProcessId = " + ((int)processId).ToString(), false);
				if (dataCheck.Rows.Count > 0) return;

				randomProcessGuid = DataCreator.GetRecordGuid("SELECT VbecsProcessGuid FROM VbecsProcess WHERE VbecsProcessId = " + ((int)processId).ToString());
			}
			DataRow drProcess = UnitTestUtil.RunSQLGetDataRow("SELECT * FROM VbecsProcess WHERE VbecsProcessGuid = '" + randomProcessGuid.ToString() + "'");

			BOL.VbecsProcess tmpObj = new BOL.VbecsProcess(drProcess);

			DataRow drNewWorkload = UnitTestUtil.RunSQLGetDataRow("SELECT TOP 1 * FROM WorkloadCache WHERE Code_Id > " + new Random().Next(1,199).ToString());

			tmpObj.AddItem(tmpObj.GetNewWorkLoad(drNewWorkload[TABLES.WorkloadCache.Lmip].ToString()));
			tmpObj.SaveChanges(Common.UpdateFunction.UnitTests);
		}

		/// <summary>
		/// Creates and returns a Modifiable BloodUnit based on input criteria
		/// </summary>
		/// <param name="modificationType">Type of modification</param>
		/// <param name="modifyMethod">Open/Closed/None</param>
		/// <param name="isCodabar">Product Indicator</param>
		/// <returns>BloodUnit</returns>
		public static BOL.BloodUnit CreateModifiableBloodUnit(Common.ProductModification modificationType, Common.ModifyUnitMethod modifyMethod, bool isCodabar)
		{
			System.Guid unitGuid = System.Guid.NewGuid();

			Guid bpGuid = GetBloodProductGuid_WithTargets(modificationType, modifyMethod, isCodabar);

			BOL.BloodUnit tmpUnit = CreateBloodUnit(unitGuid, Guid.NewGuid(), Guid.Empty, bpGuid, isCodabar);

			return(tmpUnit);
			
		}

		/// <summary>
		/// Create AboRH Confirmation tests for unit
		/// </summary>
		/// <param name="unitGuid"></param>
		/// <param name="unitAboRh"></param>
		public static void CreateAboRhConfirmationTestsForUnit(Guid unitGuid, BOL.AboRh unitAboRh)
		{
			//Create some workload
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.ABOForwardTypingUnit);
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.ABORhForwardTypingUnit);
		    Common.WorkloadProcessID workloadProcessId = unitAboRh.RH == Common.RH.Negative ? Common.WorkloadProcessID.ABORhForwardTypingUnit : Common.WorkloadProcessID.ABOForwardTypingUnit;


			DataTable rackCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 RackGuid FROM Rack WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			if (rackCheck.Rows.Count == 0)
			{
				CreateQCdRack();
			}
			Guid rackGuid = DataCreator.GetRecordGuid("SELECT TOP 1 RackGuid FROM Rack WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");			
			//Gotta update the DailyQcWorklist for Rack constructor
			UnitTestUtil.RunSQL("UPDATE DailyQcWorklist SET DailyQcExpirationDate = GETUTCDATE() WHERE RecordStatusCode = 'A' AND RackGuid = '" + rackGuid.ToString() + "'");

			BOL.BloodUnit tmpBU = new BOL.BloodUnit(unitGuid);		//ugh - Needed for UnitStatus RowVersion
			
			DataTable dtAboRh = BOL.BloodUnitTest.GetEmptyAboRhConfirmationTable();
			DataRow drData = dtAboRh.NewRow();

			drData[ARTIFICIAL.Indicator] = "V";
			drData[TABLES.BloodUnit.EyeReadableUnitId] = tmpBU.EyeReadableUnitId;

			//Set correct values
			drData[TABLES.ValidAboResult.AntiA] = (unitAboRh.Abo == Common.ABO.A  || unitAboRh.Abo == Common.ABO.AB) ? "P" : "N";
			drData[TABLES.ValidAboResult.AntiB] = (unitAboRh.Abo == Common.ABO.B  || unitAboRh.Abo == Common.ABO.AB) ? "P" : "N";
		    drData[TABLES.ValidRhResult.AntiD] =    unitAboRh.RHFactorCode;
			
			drData[TABLES.ValidAboResult.AntiAB] = "X";
			drData[TABLES.ValidRhResult.DControl] = "X";
			drData[TABLES.ValidAboResult.AboInterp] = unitAboRh.Abo.ToString();
			drData[TABLES.ValidRhResult.RhInterp] = unitAboRh.RHFactorCode;
			drData[ARTIFICIAL.AboComment] = "DataCreator.CreateAboRhConfirmationTestsForUnit()";
			drData[ARTIFICIAL.RhComment] = "DataCreator.CreateAboRhConfirmationTestsForUnit()";
			drData[TABLES.BloodUnit.BloodUnitGuid] = tmpBU.BloodUnitGuid;
			drData[TABLES.BloodUnitStatus.RowVersion] = tmpBU.BloodUnitStatus.RowVersion;
			drData[TABLES.BloodUnitStatus.BloodUnitStatusGuid] = tmpBU.BloodUnitStatus.BloodUnitStatusGuid;

			dtAboRh.Rows.Add(drData);

            BOL.BloodUnitTest.SaveAboRHBloodUnitTests(dtAboRh, DateTime.Now, Common.LogonUser.LogonUserName, "DataCreator.CreateAboRhConfirmationTestsForUnit()", true, rackGuid, Common.UpdateFunction.UnitTests, workloadProcessId, string.Empty);
		}


		public static Guid GetBloodProductGuid_WithTargets(Common.ProductModification targetModificationType, Common.ModifyUnitMethod modifyMethod, bool isCodabar)
		{
			Guid bpGuid = Guid.Empty;
			char modificationTypeChar = Common.Utility.GetProductModificationCharFromEnum(targetModificationType);
			string codabarIndicator = isCodabar ? "0" : "1";
			int methodSelect = (modifyMethod == Common.ModifyUnitMethod.Open) ? 1 : 0;


			if (_DT_ProductsWithTargets == null)
			{
				//UnitTestModifications table is created in the SQL part of AutoBuild (about 87K rows) - to run it manually, use CreateBloodUnitModfiicationXref.sql
				_DT_ProductsWithTargets = UnitTestUtil.RunSQLGetDataTable("SELECT * FROM UnitTestModifications", true);
			}
			
			//Get our subset
			System.Data.DataRow[] products = _DT_ProductsWithTargets.Select("ProductModificationCode = '" + modificationTypeChar.ToString() + "' AND FromProductCodeIndicator = " + codabarIndicator + " AND ProductModificationMethod = " + methodSelect.ToString());

			if (products.Length > 0)
			{
				int randomNum = new Random().Next(products.Length);

				bpGuid = (System.Guid)products[randomNum]["FromBloodProductGuid"];
			}
			else
			{
				throw new Exception("No BloodProducts with Targets for (" + modificationTypeChar.ToString() + "," + methodSelect.ToString() + "," + codabarIndicator + ")");
			}

			return(bpGuid);
		}

		public static void CreateAntiseraWorklistTypingRecords()
		{
			Guid awGuid = Guid.NewGuid();
			Guid awtGuid = Guid.NewGuid();

			DataTable dtRackLot = UnitTestUtil.RunSQLGetDataTable("SELECT *, RP.ReactivityPhaseText FROM RackLot RL JOIN ReactivityPhase RP on RL.ReactivityPhaseCode = RP.ReactivityPhaseCode WHERE RL.RackGuid IS NOT NULL AND RL.RecordStatusCode = 'A' AND RL.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			BOL.RackLot rl = new BOL.RackLot(dtRackLot.Rows[0]);

			DataTable dtAntigenTypingTests = BOL.AntigenTyping.GetAntigenTypingTests();

			//Logic is pulled from UC_41:DlgSelectSpecificities
			DataTable dtTmpAWT = new DataTable();

			dtTmpAWT.Columns.Add(TABLES.AntiseraWorklistTyping.AntiseraWorklistTypingGuid, typeof(Guid));
			dtTmpAWT.Columns.Add(TABLES.AntiseraWorklist.AntiseraWorklistGuid, typeof(Guid));
			dtTmpAWT.Columns.Add(TABLES.OrderableTest.OrderableTestId,typeof(string));
			dtTmpAWT.Columns.Add(TABLES.RackLot.ReagentGuid, typeof(Guid));
			dtTmpAWT.Columns.Add(TABLES.RackLot.RackGuid,typeof(Guid));
			dtTmpAWT.Columns.Add(TABLES.RackLot.ReactivityPhaseCode,typeof(string));
			dtTmpAWT.Columns.Add(TABLES.RackLot.Comments,typeof(string));
			dtTmpAWT.Columns.Add(TABLES.RackLot.DivisionCode,typeof(string));
			dtTmpAWT.Columns.Add(TABLES.RackLot.TestWithId,typeof(int));
			dtTmpAWT.Columns.Add("RackQCOverrideComment",typeof(string));
			dtTmpAWT.Columns.Add("ExpiredOverrideComment",typeof(string));
			dtTmpAWT.Columns.Add("PhaseOverrideComment",typeof(string));

			DataRow drNewAWT = dtTmpAWT.NewRow();
			drNewAWT[TABLES.AntiseraWorklistTyping.AntiseraWorklistTypingGuid] = awtGuid;
			drNewAWT[TABLES.AntiseraWorklist.AntiseraWorklistGuid] = awGuid;
			drNewAWT[TABLES.OrderableTest.OrderableTestId] = System.Convert.ToInt32(dtAntigenTypingTests.Rows[new Random().Next(dtAntigenTypingTests.Rows.Count-1)][TABLES.OrderableTest.OrderableTestId]);
			drNewAWT[TABLES.RackLot.ReagentGuid] = (Guid)dtRackLot.Rows[0][TABLES.Reagent.ReagentGuid];
			drNewAWT[TABLES.RackLot.ReactivityPhaseCode] = rl.ReactivityPhaseCode;
			drNewAWT[TABLES.RackLot.Comments] = DateTime.Now.ToString();
			drNewAWT[TABLES.RackLot.DivisionCode] = UnitTestConfig.TestDivisionCode;
			drNewAWT[TABLES.RackLot.TestWithId] = System.Convert.ToInt32(dtRackLot.Rows[0][TABLES.RackLot.TestWithId]);
			dtTmpAWT.Rows.Add(drNewAWT);

			BOL.AntiseraWorklist tmpAW = new BOL.AntiseraWorklist(awGuid);
			tmpAW.AutomatedInstrument = true;
			tmpAW.WorklistCreateDate = DateTime.Now.AddDays(-1);
			tmpAW.WorklistName = "UT:" + DateTime.Now.ToString();
			tmpAW.AutoInstrumentName = "UT:" + DateTime.Now.ToString();
			tmpAW.TestingMethodCode = Common.TestingMethod.Tube;
			
			//Get the DControl Reagent Guid - since this is a new Worklist
			DataTable dtDControl = BOL.Reagent.GetReagentsByType(UnitTestConfig.TestDivisionCode, (int)Common.WeakDTestReagents.AntiDControl);
			Guid dGuid = (Guid)dtDControl.Rows[0][TABLES.Reagent.ReagentGuid];

			DataTable dtUnits = BOL.BloodUnit.GetEmptyBloodUnitDataTable();
			Guid buGuid = DataCreator.GetRecordGuid("SELECT TOP 1 BloodUnitGuid FROM vw_BloodShipAllCurrent WHERE ComponentClassId = 1 AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			BOL.BloodUnit bu1 = new BOL.BloodUnit(buGuid);
			dtUnits.Rows.Add(bu1.LoadDataRowFromThis(dtUnits.NewRow()));

			Guid[] unitGuids = {buGuid};

			bool rc = tmpAW.Save(dtTmpAWT, dtUnits, Common.LogonUser.LogonUserName, Common.UpdateFunction.UnitTests, unitGuids, dGuid, DateTime.Now.ToString(), false);

		}

		/// <summary>
		/// Gets a random value from a table
		/// </summary>
		/// <param name="tableName"></param>
		/// <param name="filterCriteria">Coders: Don't put WHERE, just say Field = Field, append with AND for multiples</param>
		/// <returns>Random [TableName]Guid</returns>
		public static Guid GetRandomRecordGuid(string tableName, string filterCriteria)
		{
			string fieldName = tableName + "Guid";

			Guid unitGuid = Guid.Empty;
			
			filterCriteria = (filterCriteria == "") ? "" : string.Concat(" AND ", filterCriteria);
			
			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable("SELECT [" + fieldName + "] FROM [" + tableName + "] WHERE [" + fieldName + "] IS NOT NULL " + filterCriteria, true);

			if (dtTemp.Rows.Count > 0)
			{
				int randomOffset = new Random().Next(dtTemp.Rows.Count-1);
				unitGuid = (Guid) dtTemp.Rows[randomOffset][0];
			}

			return(unitGuid);
		}


		/// <summary>
		/// Gets the RecordGuid of the table
		/// </summary>
		/// <param name="sqlText"></param>
		/// <returns></returns>
		public static Guid GetRecordGuid(string sqlText)
		{
			Guid unitGuid = Guid.Empty;

			DataTable dtTemp = UnitTestUtil.RunSQLGetDataTable(sqlText, true);

			if (dtTemp.Rows.Count > 0)
			{
				unitGuid = (Guid) dtTemp.Rows[0][0];
			}

			return(unitGuid);
		}

        /// <summary>
        /// Returns a DataTable of the Patient table for a random Patient record
        /// </summary>
        /// <param name="nullMergeFields"></param>
        /// <returns></returns>
        public static DataTable GetRandomPatientRecord(bool nullMergeFields)
        {
            string whereAnds = (nullMergeFields) ? " AND FinalMergeToPatient IS NULL AND PatientMergeGroup IS NULL " : string.Empty;

            DataTable dtPatient = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM PATIENT WHERE VistaPatientId > 0 " + whereAnds + " ORDER BY LastUpdateDate");

            return (dtPatient);
        }

	    /// <summary>
		/// Override to avoid filter
		/// </summary>
		/// <param name="isCodabar"></param>
		/// <param name="componentId"></param>
		/// <returns></returns>
		public static BOL.BloodProduct GetRandomBloodProduct(bool isCodabar, Common.ComponentClass componentId)
		{
			return(GetRandomBloodProduct(isCodabar, componentId, string.Empty));
		}

		/// <summary>
		/// Returns a random BloodProductGuid from the BloodProduct table
		/// </summary>
		/// <param name="isCodabar">Codabar(TRUE) or ISBT(False)</param>
		/// <param name="componentId">ComponentClass</param>
		/// <returns>BloodProduct Object</returns>
		public static BOL.BloodProduct GetRandomBloodProduct(bool isCodabar, Common.ComponentClass componentId, string productTypes)
		{
			Guid bpGuid = Guid.Empty;
			string codabarFlag = (isCodabar) ? "0" : "1";
			string strCC = (componentId == Common.ComponentClass.ConversionClass) ? new Random().Next(1,6).ToString() : ((int)componentId).ToString();
			string strProductTypeFilter = "";

			if (productTypes != string.Empty)
			{
				strProductTypeFilter = "('" + productTypes.Replace(",", "','") + "')";
				//Remove spaces to avoid having to worry about external calling
				strProductTypeFilter = "AND ProductTypeCode IN " + strProductTypeFilter.Replace(" ", "");
			}

			//Increase performance -- only get the table once
			if (_DT_BloodProducts == null)
			{
				_DT_BloodProducts = UnitTestUtil.RunSQLGetDataTable(" SELECT BP.*, BP.ProductCodeIndicator AS BarcodeType , PT.ComponentClassId " + 
					" FROM BloodProduct BP " +
					" JOIN ProductType PT ON BP.ProductTypeCode = PT.ProductTypeCode " +
					" WHERE BP.RecordStatusCode = 'A'");
			}

			DataRow[] drRows = _DT_BloodProducts.Select("ProductCodeIndicator = " + codabarFlag + " AND ComponentClassId = " + strCC + strProductTypeFilter);

			if (drRows.Length > 0)
			{
				int randomOffset = new Random().Next(drRows.Length - 1);
				BOL.BloodProduct tmpBP = new BOL.BloodProduct(drRows[randomOffset]);
				System.Diagnostics.Debug.WriteLine("RandomBP-->" + tmpBP.ProductCode + "<");
				return(tmpBP);
			}
			else
			{
				string inputfields = "isCodabar>" + isCodabar.ToString() + "<CCId>" + componentId.ToString() + "<PTFilter>" + productTypes + "<";
				Assert.Fail("Error getting BloodProduct: " + inputfields );
				return(null);
			}
		}

		/// <summary>
		/// Grabs all Specimen UIDs from the PatientSpecimen table that start with 51*
		/// Returns the next sequential Specimen UID that would occur in the list
		/// </summary>
		/// <returns>Specimen UID as a string</returns>
		public static string GetNewSpecimenUid()
		{
			DataTable dtTest = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid LIKE '51%' ORDER BY SpecimenUid DESC");
			
			int uid = 510000000;

			foreach(DataRow  dr in dtTest.Rows)
			{

				try
				{
					uid = System.Convert.ToInt32(dr[TABLES.PatientSpecimen.SpecimenUid]);
					break;
				}
				catch
				{
					//Specimen UID most likely had an alphabetic in it... continue..
				}

			}

			uid++;		//We found the current "max", now add 1 and return that

			return(uid.ToString());
		}

		/// <summary>
		/// Insert a patient treatment record
		/// </summary>
		/// <param name="patientTreatmentGuid"></param>
		/// <param name="patientGuid"></param>
		public static void InsertPatientTreatment(Guid patientTreatmentGuid, Guid patientGuid)
		{
			string insertPatientTreatment = "INSERT INTO PATIENTTREATMENT (";
			insertPatientTreatment += TABLES.PatientTreatment.PatientTreatmentGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.PatientGuid + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateDate + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateUser + ",";
			insertPatientTreatment += TABLES.PatientTreatment.LastUpdateFunctionId + ")";
			insertPatientTreatment += "VALUES(";
			insertPatientTreatment += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientTreatment += "'" + patientGuid.ToString() + "',";
			insertPatientTreatment += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTreatment += "'" + Environment.UserName + "',";
			insertPatientTreatment += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertPatientTreatment);
		}

		/// <summary>
		/// Insert a patient order record (create a patient treatment record first)
		/// </summary>
		/// <param name="patientOrderGuid"></param>
		/// <param name="patientTreatmentGuid"></param>
		public static void InsertPatientOrder(Guid patientOrderGuid, Guid patientTreatmentGuid)
		{
			string insertPatientOrder = "INSERT INTO PATIENTORDER (";
			insertPatientOrder += TABLES.PatientOrder.PatientOrderGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.PatientTreatmentGuid + ",";
			insertPatientOrder += TABLES.PatientOrder.DivisionCode + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateDate + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateUser + ",";
			insertPatientOrder += TABLES.PatientOrder.LastUpdateFunctionId + ")";
			insertPatientOrder += "VALUES(";
			insertPatientOrder += "'" + patientOrderGuid.ToString() + "',";
			insertPatientOrder += "'" + patientTreatmentGuid.ToString() + "',";
			insertPatientOrder += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertPatientOrder += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientOrder += "'" + Environment.UserName + "',";
			insertPatientOrder += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertPatientOrder);
		}

		/// <summary>
		/// Insert a patient specimen record
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="isPreSpecimen">True for TRWs ONLY</param>
		public static void InsertPatientSpecimen(Guid patientSpecimenGuid, Guid patientGuid, bool isPreSpecimen)
		{
			string insertPatientSpecimen = "INSERT INTO PATIENTSPECIMEN (";
			insertPatientSpecimen += TABLES.PatientSpecimen.PatientSpecimenGuid + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.PatientGuid + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenUid + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenCollectionDate + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenReceivedDate + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenExpirationDate + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.SpecimenStatusCode + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.UnacceptableSpecimenIndicator + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.DivisionCode + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.LastUpdateDate + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.LastUpdateUser + ",";
			insertPatientSpecimen += TABLES.PatientSpecimen.LastUpdateFunctionId + ")";
			insertPatientSpecimen += "VALUES (";
			insertPatientSpecimen += "'" + patientSpecimenGuid.ToString() + "',";
			insertPatientSpecimen += "'" + patientGuid.ToString() + "',";
			if (isPreSpecimen)
			{
				insertPatientSpecimen += "'" + GetNewSpecimenUid() + "P',";
			}
			else
			{
				insertPatientSpecimen += "'" + GetNewSpecimenUid() + "',";
			}
			insertPatientSpecimen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientSpecimen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientSpecimen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddDays(3)) + "',";
			insertPatientSpecimen += "'R',";
			insertPatientSpecimen += "0,";
			insertPatientSpecimen += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertPatientSpecimen += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientSpecimen += "'" + Environment.UserName + "',";
			insertPatientSpecimen += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertPatientSpecimen);
		}

		/// <summary>
		/// Insert an ordered test record (create a patient order and patient specimen record first)
		/// </summary>
		/// <param name="orderedTestGuid"></param>
		/// <param name="patientOrderGuid"></param>
		/// <param name="patientSpecimenGuid"></param>
		/// <param name="orderableTest"></param>
		public static void InsertOrderedTest(Guid orderedTestGuid, Guid patientOrderGuid, Guid patientSpecimenGuid, Common.OrderableTest orderableTest)
		{
			string insertOrderedTest = "INSERT INTO ORDEREDTEST (";
			insertOrderedTest += TABLES.OrderedTest.OrderedTestGuid + ",";
			insertOrderedTest += TABLES.OrderedTest.PatientOrderGuid + ",";
			insertOrderedTest += TABLES.OrderedTest.PatientSpecimenGuid + ",";
			insertOrderedTest += TABLES.OrderedTest.OrderableTestId + ",";
			insertOrderedTest += TABLES.OrderedTest.CprsOrderNumber + ",";
			insertOrderedTest += TABLES.OrderedTest.OrderStatusCode + ",";
			insertOrderedTest += TABLES.OrderedTest.TaskStatusCode + ",";
			insertOrderedTest += TABLES.OrderedTest.DivisionCode + ",";
			insertOrderedTest += TABLES.OrderedTest.LastUpdateDate + ",";
			insertOrderedTest += TABLES.OrderedTest.LastUpdateUser + ",";
			insertOrderedTest += TABLES.OrderedTest.LastUpdateFunctionId + ")";
			insertOrderedTest += "VALUES (";
			insertOrderedTest += "'" + orderedTestGuid.ToString() + "',";
			insertOrderedTest += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedTest += "'" + patientSpecimenGuid.ToString() + "',";
			insertOrderedTest += "'" + Convert.ToInt32(orderableTest).ToString() + "',";
            insertOrderedTest += "'" + SprocHelper.GetNewCprsOrderNumber() + "',";
			insertOrderedTest += "'A',";
			insertOrderedTest += "'I',";
			insertOrderedTest += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedTest += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedTest += "'" + Environment.UserName + "',";
			insertOrderedTest += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertOrderedTest);
		}


		/// <summary>
		/// Insert and ordered component record (create a patient order record first)
		/// </summary>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="patientOrderGuid"></param>
		/// <param name="componentClass"></param>
		/// <param name="quantity"></param>
		public static void InsertOrderedComponent(Guid orderedComponentGuid, Guid patientOrderGuid, Common.ComponentClass componentClass, int quantity)
		{
			string insertOrderedComponent = "INSERT INTO ORDEREDCOMPONENT (";
			insertOrderedComponent += TABLES.OrderedComponent.OrderedComponentGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.PatientOrderGuid + ",";
			insertOrderedComponent += TABLES.OrderedComponent.ComponentClassId + ",";
			insertOrderedComponent += TABLES.OrderedComponent.CprsOrderNumber + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredUnitQuantity + ",";
			insertOrderedComponent += TABLES.OrderedComponent.RequiredDatetime + ",";
			insertOrderedComponent += TABLES.OrderedComponent.OrderStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.TaskStatusCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.DivisionCode + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateDate + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateUser + ",";
			insertOrderedComponent += TABLES.OrderedComponent.LastUpdateFunctionId + ")";
			insertOrderedComponent += "VALUES (";
			insertOrderedComponent += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedComponent += "'" + patientOrderGuid.ToString() + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(componentClass).ToString() + "',";
            insertOrderedComponent += "'" + SprocHelper.GetNewCprsOrderNumber() + "',";
			insertOrderedComponent += quantity.ToString() + ",";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'A',";
			insertOrderedComponent += "'I',";
			insertOrderedComponent += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedComponent += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedComponent += "'" + Environment.UserName + "',";
			insertOrderedComponent += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertOrderedComponent);
		}

		/// <summary>
		/// Insert an ordered unit record (create an ordered component record first)
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		public static void InsertOrderedUnit(Guid orderedUnitGuid, Guid orderedComponentGuid, Guid bloodUnitGuid)
		{
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedComponentGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.SelectedDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.SelectedTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedComponentGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertOrderedUnit);
		}

		/// <summary>
		/// Insert an ordered unit record (create an ordered component record first)
		/// </summary>
		/// <param name="orderedUnitGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		public static void InsertOrderedUnitForRepeatXMatch(Guid orderedUnitGuid, Guid orderedTestGuid, Guid bloodUnitGuid)
		{
			string insertOrderedUnit = "INSERT INTO ORDEREDUNIT (";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.OrderedTestGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.BloodUnitGuid + ",";
			insertOrderedUnit += TABLES.OrderedUnit.SelectedDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.SelectedTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchRequiredIndicator + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchResultCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.CrossmatchTechId + ",";
			insertOrderedUnit += TABLES.OrderedUnit.RecordStatusCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.DivisionCode + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateDate + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateUser + ",";
			insertOrderedUnit += TABLES.OrderedUnit.LastUpdateFunctionId + ")";
			insertOrderedUnit += "VALUES (";
			insertOrderedUnit += "'" + orderedUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + orderedTestGuid.ToString() + "',";
			insertOrderedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "1,";
			insertOrderedUnit += "'C',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'A',";
			insertOrderedUnit += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertOrderedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertOrderedUnit += "'" + Environment.UserName + "',";
			insertOrderedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertOrderedUnit);
		}

		/// <summary>
		/// Insert a blood unit test (create an ordered component and ordered unit record first)
		/// </summary>
		/// <param name="bloodUnitTestGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="orderedUnitGuid"></param>
		/// <param name="bloodTestTypeId"></param>
		/// <param name="testResultId"></param>
		public static void InsertBloodUnitTest(Guid bloodUnitTestGuid, Guid bloodUnitGuid, Guid orderedComponentGuid, Guid orderedUnitGuid, int bloodTestTypeId, string testResultId)
		{
			string insertBloodUnitTest = "INSERT INTO BLOODUNITTEST (";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestDate + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestTechId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestResultId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.EntryTechId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.EntryMethodCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.OrderedComponentGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.RecordStatusCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.DivisionCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateDate + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateUser + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			insertBloodUnitTest += "VALUES (";
			insertBloodUnitTest += "'" + bloodUnitTestGuid.ToString() + "',";
			insertBloodUnitTest += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitTest += bloodTestTypeId.ToString() + ",";
			insertBloodUnitTest += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'" + testResultId + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'M',";
			//insertBloodUnitTest += "'" + orderedComponentGuid.ToString() + "',";
			if (orderedComponentGuid != Guid.Empty)
			{
				insertBloodUnitTest += "'" + orderedComponentGuid.ToString() + "',";
			}
			else
			{
				insertBloodUnitTest += "null,";
			}


			//insertBloodUnitTest += "'" + orderedUnitGuid.ToString() + "',";
			if (orderedUnitGuid != Guid.Empty)
			{
				insertBloodUnitTest += "'" + orderedUnitGuid.ToString() + "',";
			}
			else
			{
				insertBloodUnitTest += "null,";
			}
			insertBloodUnitTest += "'2',";
			insertBloodUnitTest += "'A',";
			insertBloodUnitTest += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertBloodUnitTest += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertBloodUnitTest);
		}

		/// <summary>
		/// Insert a blood unit test related to ordered test for repeat XMatch(create an ordered test and ordered unit record first)
		/// </summary>
		/// <param name="bloodUnitTestGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="orderedComponentGuid"></param>
		/// <param name="orderedUnitGuid"></param>
		/// <param name="bloodTestTypeId"></param>
		/// <param name="testResultId"></param>
		public static void InsertBloodUnitTestForRepeatXMatch(Guid bloodUnitTestGuid, Guid bloodUnitGuid, Guid orderedTestGuid, Guid orderedUnitGuid, int bloodTestTypeId, string testResultId)
		{
			string insertBloodUnitTest = "INSERT INTO BLOODUNITTEST (";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodUnitTestGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodUnitGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.BloodTestTypeId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestDate + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestTechId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.TestResultId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.EntryTechId + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.EntryMethodCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.OrderedTestGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.OrderedUnitGuid + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.ReactivityPhaseCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.RecordStatusCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.DivisionCode + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateDate + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateUser + ",";
			insertBloodUnitTest += TABLES.BloodUnitTest.LastUpdateFunctionId + ")";
			insertBloodUnitTest += "VALUES (";
			insertBloodUnitTest += "'" + bloodUnitTestGuid.ToString() + "',";
			insertBloodUnitTest += "'" + bloodUnitGuid.ToString() + "',";
			insertBloodUnitTest += bloodTestTypeId.ToString() + ",";
			insertBloodUnitTest += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'" + testResultId + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'M',";
			//insertBloodUnitTest += "'" + orderedComponentGuid.ToString() + "',";
			if (orderedTestGuid != Guid.Empty)
			{
				insertBloodUnitTest += "'" + orderedTestGuid.ToString() + "',";
			}
			else
			{
				insertBloodUnitTest += "null,";
			}


			//insertBloodUnitTest += "'" + orderedUnitGuid.ToString() + "',";
			if (orderedUnitGuid != Guid.Empty)
			{
				insertBloodUnitTest += "'" + orderedUnitGuid.ToString() + "',";
			}
			else
			{
				insertBloodUnitTest += "null,";
			}
			insertBloodUnitTest += "'2',";
			insertBloodUnitTest += "'A',";
			insertBloodUnitTest += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertBloodUnitTest += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertBloodUnitTest += "'" + Environment.UserName + "',";
			insertBloodUnitTest += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertBloodUnitTest);
		}

		/// <summary>
		/// Insert an issued unit record (create an ordered unit record first)
		/// </summary>
		/// <param name="issuedUnitGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		public static void InsertIssuedUnit(Guid issuedUnitGuid, Guid bloodUnitGuid)
		{
			string insertIssuedUnit = "INSERT INTO ISSUEDUNIT (";
			insertIssuedUnit += TABLES.IssuedUnit.IssuedUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.BloodUnitGuid + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueDateTime + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueTechId + ",";
			insertIssuedUnit += TABLES.IssuedUnit.TransporterName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueToLocationName + ",";
			insertIssuedUnit += TABLES.IssuedUnit.InPatientLocationIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.IssueSatisfactoryIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageIndicator + ",";
			insertIssuedUnit += TABLES.IssuedUnit.RemoteStorageText + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateDate + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateUser + ",";
			insertIssuedUnit += TABLES.IssuedUnit.LastUpdateFunctionId + ")";
			insertIssuedUnit += "VALUES (";
			insertIssuedUnit += "'" + issuedUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + bloodUnitGuid.ToString() + "',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'10W P-KC',";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "1,";
			insertIssuedUnit += "0,";
			insertIssuedUnit += "'MY BEDROOM - AWW YA',";
			insertIssuedUnit += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertIssuedUnit += "'" + Environment.UserName + "',";
			insertIssuedUnit += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertIssuedUnit);
		}

		/// <summary>
		/// Insert a patient transfusion record (create an issued unit record first)
		/// </summary>
		/// <param name="patientTransfusionGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		public static void InsertPatientTransfusion(Guid patientTransfusionGuid, Guid patientGuid, Guid bloodUnitGuid)
		{
			string insertPatientTransfusion = "INSERT INTO PATIENTTRANSFUSION (";
			insertPatientTransfusion += TABLES.PatientTransfusion.PatientTransfusionGuid + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.PatientGuid + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.BloodUnitGuid + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.ReactionIndicator + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.OtherDifficultyIndicator + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.DivisionCode + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.LastUpdateDate + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.LastUpdateUser + ",";
			insertPatientTransfusion += TABLES.PatientTransfusion.LastUpdateFunctionId + ")";
			insertPatientTransfusion += "VALUES (";
			insertPatientTransfusion += "'" + patientTransfusionGuid.ToString() + "',";
			insertPatientTransfusion += "'" + patientGuid.ToString() + "',";
			insertPatientTransfusion += "'" + bloodUnitGuid.ToString() + "',";
			insertPatientTransfusion += "0,";
			insertPatientTransfusion += "0,";
			insertPatientTransfusion += "'" + Common.LogonUser.LogonUserDivisionCode + "',";
			insertPatientTransfusion += "'" + Common.VBECSDateTime.FormatDateTimeString(DateTime.Now) + "',";
			insertPatientTransfusion += "'" + Environment.UserName + "',";
			insertPatientTransfusion += "'" + Convert.ToInt32(Common.UpdateFunction.UnitTests) + "')";
			//
			UnitTestUtil.RunSQL(insertPatientTransfusion);
		}

		/// <summary>
		/// Simulates assigning a blood unit to a patient.
		/// Creates Hashtable of guids(may be used later to clear test data)
		/// Creates one specimen test record for the AboRh specified
		/// also inserts "conversion" AboRh in Patient table (for 
		/// AboRh specified):
		/// ["bloodUnitGuid", bloodUnitGuid]
		/// ["patientGuid", patientGuid]
		/// ["patientTreatmentTestOneGuid", patientTreatmentTestOneGuid]
		/// ["patientOrderTestOneGuid", patientOrderTestOneGuid]
		/// ["orderedTestTestOneGuid", orderedTestTestOneGuid]
		/// ["orderedUnitGuid", orderedUnitGuid]
		/// ["patientSpecimenTestOneGuid", patientSpecimenTestOneGuid]
		/// ["specimenTestAboTestOneGuid", specimenTestAboTestOneGuid]
		/// ["specimenTestRhTestOneGuid", specimenTestRhTestOneGuid]
		/// ["patientSpecimenTestABScreenGuid", patientSpecimenTestABScreenGuid]
		/// ["orderedComponentGuid", orderedComponentGuid]
		/// </summary>
		/// <returns>Hashtable</returns>
		public static Hashtable CreateCrossmatchedBloodUnitTestData(BOL.AboRh patientAboRh)
		{
			#region Variables

			Hashtable guidMap;

			Common.ComponentClass componentClass = Common.ComponentClass.RBC;

			bool crossmatchRequired = true;
			bool specimenRequired = true; 
			bool associateWithSpecimen = true;
		
			string specimenStatusCode = "N";	// A: In BB,  C: Converted,  N: NSR,  R: Required
			string orderUrgencyCode = "R";		// A: ASAP.  P: Pre-Op,  R: Routine,  S: Stat
			string crossmatchResultCode = "C";

			// Blood Unit
			Guid bloodUnitGuid;
			BOL.BloodUnit bloodUnit;

			// Patient
			Guid patientGuid = Guid.NewGuid();

			// Patient Treatment
			Guid patientTreatmentTestOneGuid = Guid.NewGuid();

			// Patient Order
			Guid patientOrderTestOneGuid = Guid.NewGuid();
		
			// Ordered Test
			Guid orderedTestTestOneGuid = Guid.NewGuid();

			// Ordered Unit
			Guid orderedUnitGuid = Guid.NewGuid();

			// Patient Specimen
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			
			// Specimen Tests
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			Guid patientSpecimenTestABScreenGuid = Guid.NewGuid();

			// Ordered Component
			Guid orderedComponentGuid = Guid.NewGuid();

			#endregion

			#region Create Test Data

			CreateBloodUnit(false);		//PUS code is inefficient and needs this for shipment
			bloodUnit = BOL.UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			bloodUnitGuid = bloodUnit.BloodUnitGuid;

			#region Hashtable Set-Up

			guidMap = new Hashtable();
			guidMap.Add("bloodUnitGuid", bloodUnitGuid);
			guidMap.Add("patientGuid", patientGuid);
			guidMap.Add("patientTreatmentTestOneGuid", patientTreatmentTestOneGuid);
			guidMap.Add("patientOrderTestOneGuid", patientOrderTestOneGuid);
			guidMap.Add("orderedTestTestOneGuid", orderedTestTestOneGuid);
			guidMap.Add("orderedUnitGuid", orderedUnitGuid);
			guidMap.Add("patientSpecimenTestOneGuid", patientSpecimenTestOneGuid);
			guidMap.Add("specimenTestAboTestOneGuid", specimenTestAboTestOneGuid);
			guidMap.Add("specimenTestRhTestOneGuid", specimenTestRhTestOneGuid);
			guidMap.Add("patientSpecimenTestABScreenGuid", patientSpecimenTestABScreenGuid);
			guidMap.Add("orderedComponentGuid", orderedComponentGuid);

			#endregion

			// First, we need a new Specimen Uid
			DataTable specimenUidTable = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL ORDER BY SpecimenUid DESC");
			DataColumn [] specimenUidKeys = new DataColumn []{ specimenUidTable.Columns[TABLES.PatientSpecimen.SpecimenUid] };
			specimenUidTable.PrimaryKey = specimenUidKeys;
			//
			long maxUid = 999999999999999;
			long minUid = 100000000000000;
			long specimenUid1 = minUid;
			long startIdx = minUid;
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid1 = idx;
					break;
				}
			}
			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			
			// Patient
			query.Append("INSERT INTO PATIENT (");
			query.Append(TABLES.Patient.PatientGuid + ",");
			query.Append(TABLES.Patient.PatientSsn + ",");
			query.Append(TABLES.Patient.VistaPatientId + ",");
			query.Append(TABLES.Patient.PatientLastName + ",");
			query.Append(TABLES.Patient.PatientFirstName + ",");
			query.Append(TABLES.Patient.PatientDob + ",");
			query.Append(TABLES.Patient.PatientDobCode + ",");
			query.Append(TABLES.Patient.BloodTypeCode + ",");
			query.Append(TABLES.Patient.RhFactorCode + ",");
			query.Append(TABLES.Patient.RecordStatusCode + ",");
			query.Append(TABLES.Patient.LastUpdateDate + ",");
			query.Append(TABLES.Patient.LastUpdateUser + ",");
			query.Append(TABLES.Patient.LastUpdateFunctionId + ")");
			query.Append("VALUES(");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(new Random().Next(11111,99999).ToString() + new Random().Next(1111,9999).ToString()).Append("',");
			query.Append("'").Append(SprocHelper.GetVistaPatientId().ToString()).Append("',");
			query.Append("'CreateXMatchBUTData',");
			query.Append("'SprocHelper',");
			query.Append("'").Append(DateTime.Today.AddYears(-50)).Append("',");
			query.Append("'V',");
			query.Append("'").Append(patientAboRh.Abo.ToString()).Append("',");
			query.Append("'").Append(patientAboRh.RHFactorCode).Append("',");
			query.Append("'").Append(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");

			// PatientTreatment
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// First entry into PatientOrder
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.OrderPlacedDatetime).Append(",");
			query.Append(TABLES.PatientOrder.OrderReceivedDatetime).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(DateTime.Now.AddHours(-4).ToString()).Append("',");
			query.Append("'").Append(DateTime.Now.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// OrderedTest #1
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.OrderUrgencyCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.CprsOrderNumber).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'R',");
			query.Append("'P',");
            query.Append("'" + SprocHelper.GetNewCprsOrderNumber() + "',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen #1
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.MaintainableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append(1 + ",");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#1) test: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'").Append(patientAboRh.Abo.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#1) test: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'").Append(patientAboRh.RHFactorCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: Antibody screen
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestABScreenGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'92',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// OrderedComponent
			query.Append("INSERT INTO ORDEREDCOMPONENT (");
			query.Append(TABLES.OrderedComponent.OrderedComponentGuid).Append(",");
			query.Append(TABLES.OrderedComponent.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedComponent.CprsOrderNumber).Append(",");
			query.Append(TABLES.OrderedComponent.ComponentClassId).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredUnitQuantity).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredDatetime).Append(",");
			query.Append(TABLES.OrderedComponent.SpecimenRequiredIndicator).Append(",");
			if (associateWithSpecimen) 
			{
				query.Append(TABLES.OrderedComponent.PatientSpecimenGuid).Append(",");
			}
			query.Append(TABLES.OrderedComponent.SpecimenStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderUrgencyCode).Append(",");
			query.Append(TABLES.OrderedComponent.DivisionCode).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedComponentGuid.ToString()).Append("',");		// ordered component guid
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");	// patient order guid
            query.Append("'").Append(SprocHelper.GetNewCprsOrderNumber()).Append("',");
			query.Append((int) componentClass + ",");
			query.Append(10 + ",");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (specimenRequired) 
			{
				query.Append(1 + ",");
			}
			else 
			{
				query.Append(0 + ",");							// specimen required 
			}
			if (associateWithSpecimen) 
			{
				query.Append("'" + patientSpecimenTestOneGuid + "',");	// patient specimen
			}
			query.Append("'" + specimenStatusCode + "',");		// specimen status code
			query.Append("'A',");								// order status code
			query.Append("'P',");								// task status code
			query.Append("'" + orderUrgencyCode + "',");// order urgency code
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// OrderedUnit
			query.Append("INSERT INTO ORDEREDUNIT (");
			query.Append(TABLES.OrderedUnit.OrderedUnitGuid).Append(",");
			query.Append(TABLES.OrderedUnit.OrderedComponentGuid).Append(",");
			query.Append(TABLES.OrderedUnit.BloodUnitGuid).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchDate).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchRequiredIndicator).Append(",");
			if (crossmatchResultCode != null)
			{
				query.Append(TABLES.OrderedUnit.CrossmatchResultCode).Append(",");
				query.Append(TABLES.OrderedUnit.CrossmatchTechId).Append(",");
			}
			query.Append(TABLES.OrderedUnit.RecordStatusCode).Append(",");
			query.Append(TABLES.OrderedUnit.DivisionCode).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedUnitGuid.ToString()).Append("',");
			query.Append("'").Append(orderedComponentGuid).Append("',");
			query.Append("'").Append(bloodUnitGuid).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (crossmatchRequired)	query.Append("1,");
			else query.Append("0,");
			if (crossmatchResultCode != null)
			{
				query.Append("'").Append(crossmatchResultCode).Append("',");
				query.Append("'").Append(Environment.UserName).Append("',");
			}
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Environment.UserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			Assert.IsTrue(UnitTestUtil.RunSQL(query.ToString()) > 0);
			
			#endregion

			return guidMap;
		}

		/// <summary>
		/// Simulates assigning a blood unit to a patient.
		/// Creates Hashtable of guids(may be used later to clear test data)
		/// Creates two A Pos Specimen Tests:
		/// ["bloodUnitGuid", bloodUnitGuid]
		/// ["patientGuid", patientGuid]
		/// ["patientTreatmentTestOneGuid", patientTreatmentTestOneGuid]
		/// ["patientTreatmentTestTwoGuid", patientTreatmentTestTwoGuid]
		/// ["patientOrderTestOneGuid", patientOrderTestOneGuid]
		/// ["patientOrderTestTwoGuid", patientOrderTestTwoGuid]
		/// ["orderedTestTestOneGuid", orderedTestTestOneGuid]
		/// ["orderedTestTestTwoGuid", orderedTestTestTwoGuid]
		/// ["orderedUnitGuid", orderedUnitGuid]
		/// ["patientSpecimenTestOneGuid", patientSpecimenTestOneGuid]
		/// ["patientSpecimenTestTwoGuid", patientSpecimenTestTwoGuid]
		/// ["specimenTestAboTestOneGuid", specimenTestAboTestOneGuid]
		/// ["specimenTestRhTestOneGuid", specimenTestRhTestOneGuid]
		/// ["specimenTestAboTestTwoGuid", specimenTestAboTestTwoGuid]
		/// ["specimenTestRhTestTwoGuid", specimenTestRhTestTwoGuid]
		/// ["patientSpecimenTestABScreenGuid", patientSpecimenTestABScreenGuid]
		/// ["orderedComponentGuid", orderedComponentGuid]
		/// </summary>
		/// <returns>Hashtable</returns>
		public static Hashtable CreateCrossmatchedBloodUnitTestData()
		{
			#region Variables

			Hashtable guidMap;

			Common.ComponentClass componentClass = Common.ComponentClass.RBC;

			bool crossmatchRequired = true;
			bool specimenRequired = true; 
			bool associateWithSpecimen = true;
		
			string specimenStatusCode = "N";	// A: In BB,  C: Converted,  N: NSR,  R: Required
			string orderUrgencyCode = "R";		// A: ASAP.  P: Pre-Op,  R: Routine,  S: Stat
			string crossmatchResultCode = "C";

			// Blood Unit
			Guid bloodUnitGuid;
			BOL.BloodUnit bloodUnit;

			// Patient
			Guid patientGuid = Guid.NewGuid();

			// Patient Treatment
			Guid patientTreatmentTestOneGuid = Guid.NewGuid();
			Guid patientTreatmentTestTwoGuid = Guid.NewGuid();

			// Patient Order
			Guid patientOrderTestOneGuid = Guid.NewGuid();
			Guid patientOrderTestTwoGuid = Guid.NewGuid();

			// Ordered Test
			Guid orderedTestTestOneGuid = Guid.NewGuid();
			Guid orderedTestTestTwoGuid = Guid.NewGuid();

			// Ordered Unit
			Guid orderedUnitGuid = Guid.NewGuid();

			// Patient Specimen
			Guid patientSpecimenTestOneGuid = Guid.NewGuid();
			Guid patientSpecimenTestTwoGuid = Guid.NewGuid();
			
			// Specimen Tests
			Guid specimenTestAboTestOneGuid = Guid.NewGuid();
			Guid specimenTestRhTestOneGuid = Guid.NewGuid();
			Guid specimenTestAboTestTwoGuid = Guid.NewGuid();
			Guid specimenTestRhTestTwoGuid = Guid.NewGuid();
			Guid patientSpecimenTestABScreenGuid = Guid.NewGuid();

			// Ordered Component
			Guid orderedComponentGuid = Guid.NewGuid();

			#endregion

			#region Create Test Data

			CreateBloodUnit(false);		//PUS code is inefficient and needs this for shipment
			bloodUnit = BOL.UnitTests.BOL_PatientUnitSelection.CreateBloodUnit("E002", "E0291", Common.ABO.A, Common.RH.Positive, Common.ComponentClass.RBC);
			bloodUnitGuid = bloodUnit.BloodUnitGuid;

			#region Hashtable Set-Up

			guidMap = new Hashtable();
			guidMap.Add("bloodUnitGuid", bloodUnitGuid);
			guidMap.Add("patientGuid", patientGuid);
			guidMap.Add("patientTreatmentTestOneGuid", patientTreatmentTestOneGuid);
			guidMap.Add("patientTreatmentTestTwoGuid", patientTreatmentTestTwoGuid);
			guidMap.Add("patientOrderTestOneGuid", patientOrderTestOneGuid);
			guidMap.Add("patientOrderTestTwoGuid", patientOrderTestTwoGuid);
			guidMap.Add("orderedTestTestOneGuid", orderedTestTestOneGuid);
			guidMap.Add("orderedTestTestTwoGuid", orderedTestTestTwoGuid);
			guidMap.Add("orderedUnitGuid", orderedUnitGuid);
			guidMap.Add("patientSpecimenTestOneGuid", patientSpecimenTestOneGuid);
			guidMap.Add("patientSpecimenTestTwoGuid", patientSpecimenTestTwoGuid);
			guidMap.Add("specimenTestAboTestOneGuid", specimenTestAboTestOneGuid);
			guidMap.Add("specimenTestRhTestOneGuid", specimenTestRhTestOneGuid);
			guidMap.Add("specimenTestAboTestTwoGuid", specimenTestAboTestTwoGuid);
			guidMap.Add("specimenTestRhTestTwoGuid", specimenTestRhTestTwoGuid);
			guidMap.Add("patientSpecimenTestABScreenGuid", patientSpecimenTestABScreenGuid);
			guidMap.Add("orderedComponentGuid", orderedComponentGuid);

			#endregion

			// First, we need two new Specimen Uids
			DataTable specimenUidTable = UnitTestUtil.RunSQLGetDataTable("SELECT SpecimenUid FROM PatientSpecimen WHERE SpecimenUid IS NOT NULL ORDER BY SpecimenUid DESC");
			DataColumn [] specimenUidKeys = new DataColumn []{ specimenUidTable.Columns[TABLES.PatientSpecimen.SpecimenUid] };
			specimenUidTable.PrimaryKey = specimenUidKeys;
			//
			long maxUid = 999999999999999;
			long minUid = 100000000000000;
			long specimenUid1 = minUid;
			long specimenUid2 = minUid;
			long startIdx = minUid;
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid1 = idx;
					startIdx = idx + 1;
					break;
				}
			}
			//
			for (long idx = startIdx; idx < maxUid; idx++)
			{
				if( !specimenUidTable.Rows.Contains(new object[] { idx }) )
				{ 
					specimenUid2 = idx;
					break;
				}
			}
			//
			StringBuilder query = new StringBuilder();
			//
			query.Append("BEGIN TRANSACTION T1 ");
			
			// Patient
			query.Append("INSERT INTO PATIENT (");
			query.Append(TABLES.Patient.PatientGuid + ",");
			query.Append(TABLES.Patient.PatientSsn + ",");
			query.Append(TABLES.Patient.VistaPatientId + ",");
			query.Append(TABLES.Patient.PatientLastName + ",");
			query.Append(TABLES.Patient.PatientFirstName + ",");
			query.Append(TABLES.Patient.PatientDob + ",");
			query.Append(TABLES.Patient.PatientDobCode + ",");
			query.Append(TABLES.Patient.RecordStatusCode + ",");
			query.Append(TABLES.Patient.LastUpdateDate + ",");
			query.Append(TABLES.Patient.LastUpdateUser + ",");
			query.Append(TABLES.Patient.LastUpdateFunctionId + ")");
			query.Append("VALUES(");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(new Random().Next(11111,99999).ToString() + new Random().Next(1111,9999).ToString()).Append("',");
			query.Append("'").Append(SprocHelper.GetVistaPatientId().ToString()).Append("',");
			query.Append("'CreateXMatchBUTData',");
			query.Append("'SprocHelper',");
			query.Append("'").Append(DateTime.Today.AddYears(-50)).Append("',");
			query.Append("'V',");
			query.Append("'").Append(Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Active).ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");

			// PatientTreatment
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			query.Append("INSERT INTO PATIENTTREATMENT (");
			query.Append(TABLES.PatientTreatment.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientTreatment.PatientGuid).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientTreatment.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// First entry into PatientOrder
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.OrderGroupNumber).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderingProviderFirstName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderingProviderLastName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderEnteredByFirstName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderEnteredByLastName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderingHospitalLocation).Append(",");		//new
			query.Append(TABLES.PatientOrder.InformedConsentIndicator).Append(",");		//new
			query.Append(TABLES.PatientOrder.RequestReason).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderReceivedDatetime).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderPlacedDatetime).Append(",");		//new
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(new Random().Next(11111,55555).ToString()).Append("',");		//new
			query.Append("'").Append("FIRSTPROVIDER").Append("',");		//new
			query.Append("'").Append("LASTPROVIDER").Append("',");		//new
			query.Append("'").Append("FIRSTENTERED").Append("',");		//new
			query.Append("'").Append("LASTENTERED").Append("',");		//new
			query.Append("'").Append("VA HEARTLAND - WEST").Append("',");		//new
			query.Append("0,");		//new
			query.Append("'").Append("CreateCrossmatchBloodUnitTestData()").Append("',");		//new
			query.Append("'").Append(DateTime.Now.ToString()).Append("',");		//new
			query.Append("'").Append(DateTime.Now.AddHours(-4).ToString()).Append("',");		//new
			query.Append("'").Append(patientTreatmentTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Second entry into PatientOrder (is this one needed?)
			query.Append("INSERT INTO PATIENTORDER (");
			query.Append(TABLES.PatientOrder.PatientOrderGuid).Append(",");
			query.Append(TABLES.PatientOrder.OrderGroupNumber).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderingProviderFirstName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderingProviderLastName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderEnteredByFirstName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderEnteredByLastName).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderingHospitalLocation).Append(",");		//new
			query.Append(TABLES.PatientOrder.InformedConsentIndicator).Append(",");		//new
			query.Append(TABLES.PatientOrder.RequestReason).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderReceivedDatetime).Append(",");		//new
			query.Append(TABLES.PatientOrder.OrderPlacedDatetime).Append(",");		//new
			query.Append(TABLES.PatientOrder.PatientTreatmentGuid).Append(",");
			query.Append(TABLES.PatientOrder.DivisionCode).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientOrder.LastUpdateFunctionId).Append(")");
			query.Append("VALUES(");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(new Random().Next(11111,55555).ToString()).Append("',");		//new
			query.Append("'").Append("FIRSTPROVIDER").Append("',");		//new
			query.Append("'").Append("LASTPROVIDER").Append("',");		//new
			query.Append("'").Append("FIRSTENTERED").Append("',");		//new
			query.Append("'").Append("LASTENTERED").Append("',");		//new
			query.Append("'").Append("VA HEARTLAND - WEST").Append("',");		//new
			query.Append("0,");		//new
			query.Append("'").Append("CreateCrossmatchBloodUnitTestData()").Append("',");		//new
			query.Append("'").Append(DateTime.Now.ToString()).Append("',");		//new
			query.Append("'").Append(DateTime.Now.AddHours(-4).ToString()).Append("',");		//new
			query.Append("'").Append(patientTreatmentTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			

			// Specimen #1
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.MaintainableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid1).Append("',");
			query.Append(1 + ",");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen #2
			query.Append("INSERT INTO PATIENTSPECIMEN (");
			query.Append(TABLES.PatientSpecimen.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.PatientGuid).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenCollectionDate).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.UnacceptableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.SpecimenUid).Append(",");
			query.Append(TABLES.PatientSpecimen.MaintainableSpecimenIndicator).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'0',");
			query.Append("'").Append(specimenUid2).Append("',");
			query.Append(1 + ",");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// OrderedTest #1
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestOneGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'P',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// OrderedTest #2
			query.Append("INSERT INTO ORDEREDTEST (");
			query.Append(TABLES.OrderedTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.OrderedTest.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedTest.OrderableTestId).Append(",");
			query.Append(TABLES.OrderedTest.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.OrderUrgencyCode).Append(",");
			query.Append(TABLES.OrderedTest.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedTest.CprsOrderNumber).Append(",");
			query.Append(TABLES.OrderedTest.DivisionCode).Append(",");
			query.Append(TABLES.OrderedTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedTest.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");
			query.Append("'1',");
			query.Append("'A',");
			query.Append("'R',");		//new
			query.Append("'P',");
            query.Append("'" + SprocHelper.GetNewCprsOrderNumber() + "',");		//new
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#1) test: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#1) test: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestOneGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestOneGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#2) test: A
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestAboTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'18',");
			query.Append("'A',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(1.0))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(1.0))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");
			
			// Specimen (#2) test: P
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(specimenTestRhTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'21',");
			query.Append("'P',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(1.0))).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now.AddMinutes(1.0))).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// Specimen (#2) test: Antibody screen
			query.Append("INSERT INTO SPECIMENTEST (");
			query.Append(TABLES.SpecimenTest.SpecimenTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.PatientSpecimenGuid).Append(",");
			query.Append(TABLES.SpecimenTest.OrderedTestGuid).Append(",");
			query.Append(TABLES.SpecimenTest.BloodTestTypeId).Append(",");
			query.Append(TABLES.SpecimenTest.TestResultId).Append(",");
			query.Append(TABLES.SpecimenTest.TestDate).Append(",");
			query.Append(TABLES.SpecimenTest.RecordStatusCode).Append(",");
			query.Append(TABLES.PatientSpecimen.DivisionCode).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateDate).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateUser).Append(",");
			query.Append(TABLES.PatientSpecimen.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(patientSpecimenTestABScreenGuid.ToString()).Append("',");
			query.Append("'").Append(patientSpecimenTestTwoGuid.ToString()).Append("',");
			query.Append("'").Append(orderedTestTestTwoGuid.ToString()).Append("',");
			query.Append("'92',");
			query.Append("'N',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// OrderedComponent
			query.Append("INSERT INTO ORDEREDCOMPONENT (");
			query.Append(TABLES.OrderedComponent.OrderedComponentGuid).Append(",");
			query.Append(TABLES.OrderedComponent.PatientOrderGuid).Append(",");
			query.Append(TABLES.OrderedComponent.CprsOrderNumber).Append(",");
			query.Append(TABLES.OrderedComponent.LabOrderNumber).Append(",");		//new
			query.Append(TABLES.OrderedComponent.ComponentClassId).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredUnitQuantity).Append(",");
			query.Append(TABLES.OrderedComponent.RequiredDatetime).Append(",");
			query.Append(TABLES.OrderedComponent.SpecimenRequiredIndicator).Append(",");
			if (associateWithSpecimen) 
			{
				query.Append(TABLES.OrderedComponent.PatientSpecimenGuid).Append(",");
			}
			query.Append(TABLES.OrderedComponent.SpecimenStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.TaskStatusCode).Append(",");
			query.Append(TABLES.OrderedComponent.OrderUrgencyCode).Append(",");
			query.Append(TABLES.OrderedComponent.DivisionCode).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedComponent.LastUpdateFunctionId + ")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedComponentGuid.ToString()).Append("',");		// ordered component guid
			query.Append("'").Append(patientOrderTestTwoGuid.ToString()).Append("',");	// patient order guid
            query.Append("'").Append(SprocHelper.GetNewCprsOrderNumber()).Append("',");
			query.Append("'").Append(new Random().Next(11111,55555).ToString()).Append("',");		//new
			query.Append((int) componentClass + ",");
			query.Append(10 + ",");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (specimenRequired) 
			{
				query.Append(1 + ",");
			}
			else 
			{
				query.Append(0 + ",");							// specimen required 
			}
			if (associateWithSpecimen) 
			{
				query.Append("'" + patientSpecimenTestTwoGuid + "',");	// patient specimen
			}
			query.Append("'" + specimenStatusCode + "',");		// specimen status code
			query.Append("'A',");								// order status code
			query.Append("'P',");								// task status code
			query.Append("'" + orderUrgencyCode + "',");// order urgency code
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'UnitTestUser',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("') ");
			query.Append("\n");

			// OrderedUnit
			query.Append("INSERT INTO ORDEREDUNIT (");
			query.Append(TABLES.OrderedUnit.OrderedUnitGuid).Append(",");
			query.Append(TABLES.OrderedUnit.OrderedComponentGuid).Append(",");
			query.Append(TABLES.OrderedUnit.BloodUnitGuid).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchDate).Append(",");
			query.Append(TABLES.OrderedUnit.CrossmatchRequiredIndicator).Append(",");
			if (crossmatchResultCode != null)
			{
				query.Append(TABLES.OrderedUnit.CrossmatchResultCode).Append(",");
				query.Append(TABLES.OrderedUnit.CrossmatchTechId).Append(",");
			}
			query.Append(TABLES.OrderedUnit.RecordStatusCode).Append(",");
			query.Append(TABLES.OrderedUnit.DivisionCode).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateDate).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateUser).Append(",");
			query.Append(TABLES.OrderedUnit.LastUpdateFunctionId).Append(")");
			query.Append("VALUES (");
			query.Append("'").Append(orderedUnitGuid.ToString()).Append("',");
			query.Append("'").Append(orderedComponentGuid).Append("',");
			query.Append("'").Append(bloodUnitGuid).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			if (crossmatchRequired)	query.Append("1,");
			else query.Append("0,");
			if (crossmatchResultCode != null)
			{
				query.Append("'").Append(crossmatchResultCode).Append("',");
				query.Append("'").Append(Environment.UserName).Append("',");
			}
			query.Append("'A',");
			query.Append("'").Append(Common.LogonUser.LogonUserDivisionCode).Append("',");
			query.Append("'").Append(Common.VBECSDateTime.FormatDateTimeString(DateTime.Now)).Append("',");
			query.Append("'").Append(Environment.UserName).Append("',");
			query.Append("'").Append(Convert.ToInt32(Common.UpdateFunction.UnitTests)).Append("')");
			//
			query.Append("COMMIT TRANSACTION T1 ");
			//
			Assert.IsTrue(UnitTestUtil.RunSQL(query.ToString()) > 0);
			
			#endregion

			return guidMap;
		}


		/// <summary>
		/// Creates an autologous BloodUnit and Shipment Record
		/// </summary>
		/// <param name="unitGuid"></param>
		/// <param name="statusGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="bloodProuctGuid"></param>
		/// <returns></returns>
		public static void CreateAutologousBloodUnit(Guid unitGuid, Guid statusGuid, Guid patientGuid, Guid bloodProuctGuid)
		{
			BOL.BloodProduct tstBP = new BOL.BloodProduct(bloodProuctGuid);
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid, bloodProuctGuid);
		
			BOL.Division tmpDiv = new BOL.Division(UnitTestConfig.TestDivisionCode);
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(tmpDiv.LocalSupplierGuid, false);

			//Shipment now requires ProductShipperGuid, so, we're going to create ProductShipper(if not one already)
			BOL.ProductShipper tmpPS = tstBP.ProductShippers.GetElementByLocalSupplierGuid(tmpLS.LocalSupplierGuid);
			if (tmpPS == null)
			{
				tmpPS = new BOL.ProductShipper(Guid.NewGuid(), bloodProuctGuid);
				tmpPS.ProductCost = 1;
				tmpPS.LocalSupplierGuid = tmpLS.LocalSupplierGuid;
				tmpPS.ReturnCreditPercentage = 1;
				tmpPS.RecordStatusCode = Common.RecordStatusCode.Active;
				tstBP.ProductShippers.Add(tmpPS);
				tstBP.Save(Common.UpdateFunction.UnitTests);
			}

			BOL.IncomingShipment tmpShipment;

			DataTable dtS = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Shipment WHERE InvoiceNumber = 'UnitTests' AND LocalSupplierGuid = '" + tmpLS.LocalSupplierGuid.ToString() + "'");
			if (dtS.Rows.Count == 0)
			{
				tmpShipment = new BOL.IncomingShipment("UnitTests", tmpLS.LocalSupplierGuid, tmpPS.ProductShipperGuid,  DateTime.Now);
				tmpShipment.LocalSupplier = tmpLS;
			}
			else
			{
				tmpShipment = new BOL.IncomingShipment((Guid)dtS.Rows[0][TABLES.Shipment.ShipmentGuid]);
			}

			//SET THIS HERE, CUZ INCOMINGINSHPMENT DOESN't LOAD FOR EXISTING SHIPMENTS -- ????
			tmpShipment.ProductShipper = tmpPS;

			//Set our BloodUnit properties
			tmpUnit.LabelTypeCode = "C";
			tmpUnit.UnitFdaRegistrationNumber = tmpLS.FDARegistrationNumber;

			string unitId = new Random().Next(11111,99999).ToString() + new Random().Next(11111,99999).ToString();
			tmpUnit.SetLabelUnitId(unitId, false);
			tmpUnit.SetEyeReadableUnitId(unitId, false);
			
			tmpUnit.BloodProductGuid = bloodProuctGuid;
			tmpUnit.DonationType = new BOL.DonationType(9);
			tmpUnit.OriginalVolume = 420;

			tmpUnit.EntryMethodCode = "M";
			tmpUnit.RecordStatusCode = Common.RecordStatusCode.Active;
			tmpUnit.BloodUnitFinance.UnitCost = 98.76M;
			tmpUnit.BloodUnitFinance.SpecialTestingCost = 54.32M;
			tmpUnit.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(7);
			tmpUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			tmpUnit.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			tmpUnit.BloodUnitMedia.DirectedPatientGuid = patientGuid;

			tmpUnit.Comments = "Entered VIA CreateBloodUnit()";

			tmpUnit.BloodUnitStatus.BloodUnitStatusGuid = statusGuid;
	
			System.Collections.ArrayList emptyAL = new System.Collections.ArrayList();
			string bloodUnitInfo = string.Empty;
			try
			{
				DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogin);
				tmpShipment.SaveIncomingShipment(tmpUnit, false, emptyAL, emptyAL, Common.UpdateFunction.UnitTests , bloodUnitInfo, Common.WorkloadProcessID.UnitLogin);
			}
			catch(Exception err)
			{
				throw new BOL.BusinessObjectException("Create failed! Reason:" + err.Message);
			}

		}

		/// <summary>
		/// Creates a directed BloodUnit and Shipment Record
		/// </summary>
		/// <param name="unitGuid"></param>
		/// <param name="statusGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="bloodProuctGuid"></param>
		/// <returns></returns>
		public static void CreateDirectedBloodUnit(Guid unitGuid, Guid statusGuid, Guid patientGuid, Guid bloodProuctGuid)
		{
			BOL.BloodProduct tstBP = new BOL.BloodProduct(bloodProuctGuid);
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid, bloodProuctGuid);
		
			BOL.Division tmpDiv = new BOL.Division(UnitTestConfig.TestDivisionCode);
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(tmpDiv.LocalSupplierGuid, false);

			//Shipment now requires ProductShipperGuid, so, we're going to create ProductShipper(if not one already)
			BOL.ProductShipper tmpPS = tstBP.ProductShippers.GetElementByLocalSupplierGuid(tmpLS.LocalSupplierGuid);
			if (tmpPS == null)
			{
				tmpPS = new BOL.ProductShipper(Guid.NewGuid(), bloodProuctGuid);
				tmpPS.ProductCost = 1;
				tmpPS.LocalSupplierGuid = tmpLS.LocalSupplierGuid;
				tmpPS.ReturnCreditPercentage = 1;
				tmpPS.RecordStatusCode = Common.RecordStatusCode.Active;
				tstBP.ProductShippers.Add(tmpPS);
				tstBP.Save(Common.UpdateFunction.UnitTests);
			}

			BOL.IncomingShipment tmpShipment;

			DataTable dtS = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Shipment WHERE InvoiceNumber = 'UnitTests' AND LocalSupplierGuid = '" + tmpLS.LocalSupplierGuid.ToString() + "'");
			if (dtS.Rows.Count == 0)
			{
				tmpShipment = new BOL.IncomingShipment("UnitTests", tmpLS.LocalSupplierGuid, tmpPS.ProductShipperGuid,  DateTime.Now);
				tmpShipment.LocalSupplier = tmpLS;
			}
			else
			{
				tmpShipment = new BOL.IncomingShipment((Guid)dtS.Rows[0][TABLES.Shipment.ShipmentGuid]);
			}

			//SET THIS HERE, CUZ INCOMINGINSHPMENT DOESN't LOAD FOR EXISTING SHIPMENTS -- ????
			tmpShipment.ProductShipper = tmpPS;

			//Set our BloodUnit properties
			tmpUnit.LabelTypeCode = "C";
			tmpUnit.UnitFdaRegistrationNumber = tmpLS.FDARegistrationNumber;

			string unitId = new Random().Next(11111,99999).ToString() + new Random().Next(11111,99999).ToString();
			tmpUnit.SetLabelUnitId(unitId, false);
			tmpUnit.SetEyeReadableUnitId(unitId, false);
			
			tmpUnit.BloodProductGuid = bloodProuctGuid;
			tmpUnit.DonationType = new BOL.DonationType(12);
			tmpUnit.OriginalVolume = 420;

			tmpUnit.EntryMethodCode = "M";
			tmpUnit.RecordStatusCode = Common.RecordStatusCode.Active;
			tmpUnit.BloodUnitFinance.UnitCost = 98.76M;
			tmpUnit.BloodUnitFinance.SpecialTestingCost = 54.32M;
			tmpUnit.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(7);
			tmpUnit.BloodUnitMedia.AboRh.Abo = Common.ABO.A;
			tmpUnit.BloodUnitMedia.AboRh.RH = Common.RH.Positive;
			tmpUnit.BloodUnitMedia.DirectedPatientGuid = patientGuid;

			tmpUnit.Comments = "Entered VIA CreateBloodUnit()";

			tmpUnit.BloodUnitStatus.BloodUnitStatusGuid = statusGuid;
	
			System.Collections.ArrayList emptyAL = new System.Collections.ArrayList();
			string bloodUnitInfo = string.Empty;
			
			try
			{
				DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogin);
				tmpShipment.SaveIncomingShipment(tmpUnit, false, emptyAL, emptyAL, Common.UpdateFunction.UnitTests , bloodUnitInfo, Common.WorkloadProcessID.UnitLogin);
			}
			catch(Exception err)
			{
				throw new BOL.BusinessObjectException("Create failed! Reason:" + err.Message);
			}

		}

		/// <summary>
		/// Creates a BloodUnit and Shipment Record - nothing is retrned to the user
		/// </summary>
		/// <param name="isCodabar"></param>
		public static BOL.BloodUnit CreateRBCBloodUnit(bool isCodabar)
		{
			BOL.BloodUnit tmpUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, GetRandomBloodProduct(isCodabar, Common.ComponentClass.RBC, "").BloodProductGuid , isCodabar);
			return(tmpUnit);
		}

		/// <summary>
		/// Creates a BloodUnit and Shipment Record - nothing is retrned to the user
		/// </summary>
		/// <param name="isCodabar"></param>
		public static BOL.BloodUnit CreateBloodUnit(bool isCodabar)
		{
			BOL.BloodUnit tmpUnit = DataCreator.CreateBloodUnit(Guid.NewGuid(), Guid.NewGuid(), Guid.Empty, GetRandomBloodProduct(isCodabar, Common.ComponentClass.ConversionClass, "").BloodProductGuid , isCodabar);
			return(tmpUnit);
		}

		/// <summary>
		/// Creates a BloodUnit and Shipment Record
		/// </summary>
		/// <param name="unitGuid"></param>
		/// <param name="statusGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="bloodProuctGuid"></param>
		/// <returns></returns>
		public static BOL.BloodUnit CreateBloodUnit(Guid unitGuid, Guid statusGuid, Guid patientGuid, Guid bloodProductGuid, bool isCodabar)
		{
			//Good ol' Random Fun!
			BOL.AboRh tmpAboRh = new BOL.AboRh();

			int ranAbo = new Random().Next(1,4);
			int ranRh = new Random().Next(1,2);

			if (ranAbo == 1) tmpAboRh.Abo = Common.ABO.A;
			if (ranAbo == 2) tmpAboRh.Abo = Common.ABO.B;
			if (ranAbo == 3) tmpAboRh.Abo = Common.ABO.AB;
			if (ranAbo == 4) tmpAboRh.Abo = Common.ABO.O;
			
			tmpAboRh.RH = (ranRh == 1) ? Common.RH.Positive : Common.RH.Negative;

			BOL.BloodUnit tmpUnit = DataCreator.CreateBloodUnit(unitGuid, statusGuid, patientGuid, bloodProductGuid, tmpAboRh.Abo, tmpAboRh.RH, UnitTestConfig.TestDivisionCode, isCodabar);
			return(tmpUnit);
		}


		/// <summary>
		/// Creates a BloodUnit and Shipment Record
		/// </summary>
		/// <param name="unitGuid"></param>
		/// <param name="statusGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="bloodProuctGuid"></param>
		/// <param name="isCodabar"></param>
		/// <returns></returns>
		public static BOL.BloodUnit CreateBloodUnit(Guid unitGuid, Guid statusGuid, Guid patientGuid, Guid bloodProductGuid, Common.ABO abo, Common.RH rh, string divisionCode, bool isCodabar)
		{
			BOL.BloodProduct tstBP = new BOL.BloodProduct(bloodProductGuid);
			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(unitGuid, bloodProductGuid);
		
			BOL.Division tmpDiv = new BOL.Division(divisionCode);
			BOL.LocalSupplier tmpLS = new BOL.LocalSupplier(tmpDiv.LocalSupplierGuid, false);
			
			if (tmpLS.RecordStatusCode == Common.RecordStatusCode.Inactive)
			{
				UnitTestUtil.RunSQL("UPDATE LocalSupplier SET RecordStatusCode = 'L' WHERE LocalSupplierGuid = '" + tmpLS.LocalSupplierGuid.ToString() + "'");
				tmpLS = new gov.va.med.vbecs.BOL.LocalSupplier(tmpDiv.LocalSupplierGuid, false);
			}

			//Shipment now requires ProductShipperGuid, so, we're going to create ProductShipper(if not one already)
			BOL.ProductShipper tmpPS = tstBP.ProductShippers.GetElementByLocalSupplierGuid(tmpLS.LocalSupplierGuid);
			if (tmpPS == null)
			{
				tmpPS = new BOL.ProductShipper(Guid.NewGuid(), bloodProductGuid);
				tmpPS.ProductCost = 1;
				tmpPS.LocalSupplierGuid = tmpLS.LocalSupplierGuid;
				tmpPS.ReturnCreditPercentage = 1;
				tmpPS.RecordStatusCode = Common.RecordStatusCode.Active;
				tstBP.ProductShippers.Add(tmpPS);
				tstBP.Save(Common.UpdateFunction.UnitTests);
			}

			if (BOL.BloodProduct.IsBloodProductActive(bloodProductGuid))
			{
				BOL.BloodProduct updateBP = new BOL.BloodProduct(bloodProductGuid);
				DataTable dtCodes = UnitTestUtil.RunSQLGetDataTable("SELECT " + TABLES.HcpcsCache.HcpcsCode + " FROM " + TABLES.HcpcsCache.TableName);
				if (dtCodes.Rows.Count == 0)
				{
					VistALinkUnitTestHarness.RequireConnection();
					DAL.VAL.VistaCache.UpdateCache(-1);		//Put the cache data in
					VistALinkUnitTestHarness.ReleaseConnection();
					dtCodes = UnitTestUtil.RunSQLGetDataTable("SELECT " + TABLES.HcpcsCache.HcpcsCode + " FROM " + TABLES.HcpcsCache.TableName, true);
				}
				updateBP.HcpcsCode = dtCodes.Rows[new Random().Next(dtCodes.Rows.Count-1)][TABLES.HcpcsCache.HcpcsCode].ToString();
				updateBP.Save(Common.UpdateFunction.UnitTests);		//Save the Hcpcs
			}
			else
			{
				throw new Exception("Product Should Have a Shipper.. Wierdness!");
			}

			BOL.IncomingShipment tmpShipment;

			DataTable dtS = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM Shipment WHERE InvoiceNumber = 'UnitTests' AND LocalSupplierGuid = '" + tmpLS.LocalSupplierGuid.ToString() + "' AND RecordStatusCode = 'A'");
			if (dtS.Rows.Count == 0)
			{
				tmpShipment = new BOL.IncomingShipment("UnitTests", tmpLS.LocalSupplierGuid, tmpPS.ProductShipperGuid,  DateTime.Now.AddDays(-1));
				tmpShipment.LocalSupplier = tmpLS;
				tmpShipment.ShipmentDate = DateTime.Now.AddDays(-1);		//ShipmentDate must be entered manually, constructor ignores value, use past date since this all happens so fast, that's what she said!
			}
			else
			{
				tmpShipment = new BOL.IncomingShipment((Guid)dtS.Rows[0][TABLES.Shipment.ShipmentGuid]);
			}

			//SET THIS HERE, CUZ INCOMINGINSHPMENT DOESN't LOAD FOR EXISTING SHIPMENTS -- ????
			tmpShipment.ProductShipper = tmpPS;
				
			bool unitNotExists = false;
			//Set our BloodUnit properties
			if (isCodabar)
			{
				while (!unitNotExists)
				{
					tmpUnit.LabelTypeCode = "C";
					tmpUnit.UnitFdaRegistrationNumber = tmpLS.FDARegistrationNumber;

					string codabarId = new Random().Next(11111,99998).ToString() + new Random().Next(11112,99999).ToString();
					tmpUnit.SetLabelUnitId(codabarId, false);
					tmpUnit.SetEyeReadableUnitId(codabarId, false);

					DataTable dtExists = UnitTestUtil.RunSQLGetDataTable("SELECT BloodUnitGuid FROM BloodUnit WHERE EyeReadableUnitId = '" + codabarId + "' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND RecordStatusCode = 'A' AND BloodProductGuid = '" + tstBP.BloodProductGuid.ToString() + "'");
					unitNotExists = (dtExists.Rows.Count == 0);
				}
			}
			else
			{
				while (!unitNotExists)
				{
					tmpUnit.LabelTypeCode = "I";
					tmpUnit.UnitFdaRegistrationNumber = tmpLS.FDARegistrationNumber;

					string isbtUnitId = "T" + new Random().Next(100000,999999).ToString() + new Random().Next(100000,999999).ToString();
					tmpUnit.SetLabelUnitId(isbtUnitId, false);
					tmpUnit.SetEyeReadableUnitId(isbtUnitId, false);
					tmpUnit.UnitDivisionId = "00";

					DataTable dtExists = UnitTestUtil.RunSQLGetDataTable("SELECT BloodUnitGuid FROM BloodUnit WHERE EyeReadableUnitId = '" + isbtUnitId + "' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' AND RecordStatusCode = 'A' AND BloodProductGuid = '" + tstBP.BloodProductGuid.ToString() + "'");
					unitNotExists = (dtExists.Rows.Count == 0);
				}
			}


			tmpUnit.BloodProductGuid = bloodProductGuid;
			tmpUnit.DonationType = new BOL.DonationType(2);
			tmpUnit.OriginalVolume = 420;

			tmpUnit.EntryMethodCode = "M";
			tmpUnit.RecordStatusCode = Common.RecordStatusCode.Active;
			tmpUnit.BloodUnitFinance.UnitCost = 98.76M;
			tmpUnit.BloodUnitFinance.SpecialTestingCost = 54.32M;
			tmpUnit.BloodUnitMedia.UnitExpirationDate = DateTime.Now.AddDays(7);
			tmpUnit.BloodUnitMedia.AboRh.Abo = abo;
			tmpUnit.BloodUnitMedia.AboRh.RH = rh;

			if (patientGuid != Guid.Empty)
				tmpUnit.BloodUnitMedia.DirectedPatientGuid = patientGuid;

			tmpUnit.Comments = "DataCreator.CreateBloodUnit()";

			tmpUnit.BloodUnitStatus.BloodUnitStatusGuid = statusGuid;
	
			System.Collections.ArrayList emptyAL = new System.Collections.ArrayList();
			string bloodUnitInfo = string.Empty;
			
			try
			{
				DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogin);
				tmpShipment.SaveIncomingShipment(tmpUnit, false, emptyAL, emptyAL, Common.UpdateFunction.UnitTests , bloodUnitInfo, Common.WorkloadProcessID.UnitLogin);
				return(new BOL.BloodUnit(tmpUnit.BloodUnitGuid));		//Return a new one so that all the data is pulled back.. UnitStatusCode wasn't there without this
			}
			catch(Exception err)
			{
				throw new BOL.BusinessObjectException("Create failed! Reason:" + err.Message);
			}

		}

		/// <summary>
		/// Creates a Local Supplier
		/// </summary>
		public static BOL.LocalSupplier CreateLocalSupplier()
		{
			Guid facilityGuid = GetRandomRecordGuid(TABLES.Facility.TableName, "FacilityGuid NOT IN (SELECT FacilityGuid FROM LocalSupplier WHERE FacilityGuid IS NOT NULL) AND IccbbaRegistrationNumber IS NOT NULL AND AddressGuid IS NOT NULL");
			BOL.FacilityDefinition tmpFacility = BOL.FacilityDefinition.GetByGuid(facilityGuid);

			BOL.LocalSupplier newSupplier = new BOL.LocalSupplier(tmpFacility.FdaRegistrationNumber, false);
			newSupplier.FacilityGuid = facilityGuid;
			newSupplier.FacilityPrefix = new Random().Next(11,99).ToString();
			newSupplier.IsCollectionFacility = true;
			newSupplier.RecordStatusCode = Common.RecordStatusCode.LocalEntry;

			newSupplier.InsertLocalSupplier(Common.UpdateFunction.UnitTests);

			return(newSupplier);
		}


		/// <summary>
		/// Creates a BloodUnitTest
		/// </summary>
		/// <param name="confWorklistGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="testTypeId"></param>
		/// <param name="testResult"></param>
		/// <returns></returns>
		public static void CreateBloodUnitTest(Guid confWorklistGuid, Guid bloodUnitGuid, int testTypeId, string testResult)
		{
			Guid commentGuid = Guid.NewGuid();

			BOL.BloodUnitTest tmpBUT = new BOL.BloodUnitTest(bloodUnitGuid, testTypeId);

			tmpBUT.TestResultId = testResult;
			tmpBUT.EntryMethodCode = "M";
			tmpBUT.TestComments = commentGuid.ToString();
			//tmpBUT.Rack = new BOL.Rack(rackGuid);

			tmpBUT.SaveBloodUnitTest(confWorklistGuid, Common.UpdateFunction.UnitTests);

		}

		/// <summary>
		/// Creates a IssuedUnit
		/// </summary>
		/// <returns></returns>
		public static void CreateIssuedBloodUnit(bool createOrderStatus)
		{
			//There BETTER be Patient Data, or else!
			Guid patientGuid = GetRecordGuid("SELECT TOP 1 P.PatientGuid FROM Patient P JOIN PatientTreatment PT ON P.PatientGuid = PT.PatientGuid WHERE FinalMergeToPatient IS NULL ");
			Guid ptGuid = GetRecordGuid("SELECT TOP 1 PatientTreatmentGuid FROM PatientTreatment WHERE PatientGuid = '" + patientGuid.ToString() + "'");
			
			DataTable dtCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodProductGuid FROM ProductShipper WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			if (dtCheck.Rows.Count == 0) CreateBloodUnit(true);	//lazy but fast
			Guid bpGuid = GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM ProductShipper WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			Guid busGuid = Guid.NewGuid();
			Guid buGuid = Guid.NewGuid();
			Guid iuGuid = Guid.NewGuid();
			Guid ouGuid = Guid.NewGuid();
			Guid ocGuid = Guid.NewGuid();
			Guid poGuid = Guid.NewGuid();
			Guid psGuid = Guid.NewGuid();
			Guid otGuid = Guid.NewGuid();
			string specimenUid = GetNewSpecimenUid();

			//Create a unit to Issue  --- Was manullay setting UnitStatusCode to I here... UGH!
			CreateBloodUnit(buGuid, busGuid, Guid.Empty, bpGuid, true);

			//Hopefully that worked...
			//Not sure if I need a PatientSpecimen or not, so holding off here....
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();

			dtArray.Add(SprocHelper.GetInsertPatientSpecimenSprocTable(psGuid, patientGuid, specimenUid));
			spArray.Add(SPROCS.InsertPatientSpecimen.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertPatientOrderSprocTable(poGuid, ptGuid));
			spArray.Add(SPROCS.InsertPatientOrder.StoredProcName);

			dtArray.Add(SprocHelper.GetHL7UpdateOrderedComponent(ocGuid, poGuid, createOrderStatus));
			spArray.Add(SPROCS.HL7UpdateOrderedComponent.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertOrderedUnitSprocTable(ouGuid, ocGuid, buGuid));
			spArray.Add(SPROCS.InsertOrderedUnit.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertIssuedUnitSprocTable(iuGuid, buGuid, patientGuid));
			spArray.Add(SPROCS.InsertIssuedUnit.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertOrderedTestSprocTable(otGuid, poGuid, psGuid));
			spArray.Add(SPROCS.InsertOrderedTest.StoredProcName);

			int retValue = new Common.StoredProcedure().TransactionalGetValue(spArray, dtArray);

			if (retValue != 0)
			{
				throw new BOL.BusinessObjectException("CRAP! THE ISSUE UNIT CREATION FAILED!!!");
			}
			else
			{
				if (createOrderStatus)
				{
					UnitTestUtil.RunSQL("UPDATE OrderedComponent SET TaskStatusCode = 'I' WHERE OrderedComponentGuid = '" + ocGuid.ToString() + "'");
				}
			}
		}

		/// <summary>
		/// Creates a IssuedUnit
		/// </summary>
		/// <returns></returns>
		public static Guid CreateIssuedBloodUnit()
		{
			//There BETTER be Patient Data, or else!
			Guid patientGuid = GetRecordGuid("SELECT TOP 1 P.PatientGuid FROM Patient P JOIN PatientTreatment PT ON P.PatientGuid = PT.PatientGuid");
			Guid ptGuid = GetRecordGuid("SELECT TOP 1 PatientTreatmentGuid FROM PatientTreatment WHERE PatientGuid = '" + patientGuid.ToString() + "'");
			
			DataTable dtCheck = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 BloodProductGuid FROM ProductShipper WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			if (dtCheck.Rows.Count == 0) CreateBloodUnit(true);	//lazy but fast
			Guid bpGuid = GetRecordGuid("SELECT TOP 1 BloodProductGuid FROM ProductShipper WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			Guid busGuid = Guid.NewGuid();
			Guid buGuid = Guid.NewGuid();
			Guid iuGuid = Guid.NewGuid();
			Guid ouGuid = Guid.NewGuid();
			Guid ocGuid = Guid.NewGuid();
			Guid poGuid = Guid.NewGuid();
			Guid psGuid = Guid.NewGuid();
			Guid otGuid = Guid.NewGuid();
			string specimenUid = GetNewSpecimenUid();

			//Create a unit to Issue  --- Was manullay setting UnitStatusCode to I here... UGH!
			CreateBloodUnit(buGuid, busGuid, Guid.Empty, bpGuid, true);

			//Hopefully that worked...
			//Not sure if I need a PatientSpecimen or not, so holding off here....
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();
			System.Collections.ArrayList spArray = new System.Collections.ArrayList();

			dtArray.Add(SprocHelper.GetInsertPatientSpecimenSprocTable(psGuid, patientGuid, specimenUid));
			spArray.Add(SPROCS.InsertPatientSpecimen.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertPatientOrderSprocTable(poGuid, ptGuid));
			spArray.Add(SPROCS.InsertPatientOrder.StoredProcName);

			dtArray.Add(SprocHelper.GetHL7UpdateOrderedComponent(ocGuid, poGuid, true));
			spArray.Add(SPROCS.HL7UpdateOrderedComponent.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertOrderedUnitSprocTable(ouGuid, ocGuid, buGuid));
			spArray.Add(SPROCS.InsertOrderedUnit.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertIssuedUnitSprocTable(iuGuid, buGuid, patientGuid));
			spArray.Add(SPROCS.InsertIssuedUnit.StoredProcName);

			dtArray.Add(SprocHelper.GetInsertOrderedTestSprocTable(otGuid, poGuid, psGuid));
			spArray.Add(SPROCS.InsertOrderedTest.StoredProcName);

			int retValue = new Common.StoredProcedure().TransactionalGetValue(spArray, dtArray);

			if (retValue != 0)
			{
				throw new BOL.BusinessObjectException("CRAP! THE ISSUE UNIT CREATION FAILED!!!");
			}
			else
			{
				UnitTestUtil.RunSQL("UPDATE OrderedComponent SET TaskStatusCode = 'I' WHERE OrderedComponentGuid = '" + ocGuid.ToString() + "'");
			}
			//
			return iuGuid;
		}

		public static System.Guid CreateOutgoingShipment(System.Guid bloodUnitGuid, bool confirmed)
		{
			Guid lsGuid = DataCreator.GetRecordGuid("SELECT LocalSupplierGuid FROM VamcDivision WHERE DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");

			BOL.BloodUnit tmpUnit = new BOL.BloodUnit(bloodUnitGuid);
			
			BOL.OutgoingShipmentCollection tmpOSC = BOL.OutgoingShipmentCollection.CreateOutgoingInvoice("DC.CreateOS" + new Random().Next(500).ToString(), lsGuid, DateTime.Now.AddDays(-1), Common.ShipmentType.ReturnToSupplier, Common.LogonUser.LogonUserName, Common.UpdateFunction.UnitTests);
			tmpOSC.AddUnitToInvoice(BOL.OutgoingShipmentCollection.LineItem.GetLineItem(tmpUnit.BloodUnitGuid));
			tmpOSC.SaveChangesToInvoiceItems(new System.Collections.ArrayList(), Common.UpdateFunction.UnitTests);

			System.Guid shipmentGuid = DataCreator.GetRecordGuid("SELECT BUS.OutgoingShipmentGuid FROM BloodUnitShipment BUS JOIN BloodUnit BU ON BU.BloodUnitGuid = BUS.BloodUnitGUID AND BUS.RecordStatusCode = 'A' AND BU.DivisionCode = '" + UnitTestConfig.TestDivisionCode + "' WHERE OutgoingShipmentGuid IS NOT NULL AND BU.BloodUnitGuid = '" + tmpUnit.BloodUnitGuid.ToString() + "'");

			if (confirmed)
			{
				tmpOSC.Confirmed = false;
				tmpOSC.ReadyToConfirm = true;

				DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.UnitLogout);
				tmpOSC.ConfirmShipment(DateTime.Now, Common.UpdateFunction.UnitTests, Common.WorkloadProcessID.UnitLogout);

			}
			return(shipmentGuid);
		}

		/// <summary>
		/// Creates a patient Assignment for the BloodUnit
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="existingPatient"></param>
		/// <returns></returns>
		public static void CreatePatientAssignment(Guid bloodUnitGuid, Guid patientGuid, bool existingPatient)
		{
			Guid ptGuid = Guid.NewGuid();
			Guid poGuid = Guid.NewGuid();
			Guid ocGuid = Guid.NewGuid();
			Guid ouGuid = Guid.NewGuid();

            System.Collections.ArrayList dtArray = new ArrayList();
			System.Collections.ArrayList sprocArray = new ArrayList();

			//Create a brand new patient to build off of so we don't mess with existing data
            if (!existingPatient)
            {
                DataTable dtPatient = SprocHelper.GetInsertPatientSprocTable(patientGuid);
                dtArray.Add(dtPatient);
                sprocArray.Add(SPROCS.InsertPatient.StoredProcName);
            }

		    DataTable dtPatientTreatment = SprocHelper.GetInsertPatientTreatmentSprocTable(ptGuid, patientGuid);
			DataTable dtPatientOrder = SprocHelper.GetInsertPatientOrderSprocTable(poGuid, ptGuid);
			DataTable dtOrderedComponent = SprocHelper.GetHL7UpdateOrderedComponent(ocGuid, poGuid, false);
			DataTable dtOrderedUnit = SprocHelper.GetInsertOrderedUnitSprocTable(ouGuid, ocGuid, bloodUnitGuid);

			dtArray.Add(dtPatientTreatment);
			sprocArray.Add(SPROCS.InsertPatientTreatment.StoredProcName);

			dtArray.Add(dtPatientOrder);
			sprocArray.Add(SPROCS.InsertPatientOrder.StoredProcName);

			dtArray.Add(dtOrderedComponent);
			sprocArray.Add(SPROCS.HL7UpdateOrderedComponent.StoredProcName);

			dtArray.Add(dtOrderedUnit);
			sprocArray.Add(SPROCS.InsertOrderedUnit.StoredProcName);

			int retValue = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);


			Assert.AreEqual(0, retValue, "SprocHelper(" + bloodUnitGuid.ToString() + ")");
		}

		public static void CreateNewOrder(bool newPatient)
		{
			Guid orderedComponentGuid = Guid.NewGuid();
			Guid orderedTestGuid = Guid.NewGuid();
			Guid patientOrderGuid = Guid.NewGuid();
			
			Guid patientTreatmentGuid = Guid.NewGuid();
			Guid patientGuid = newPatient ? Guid.NewGuid() : GetRandomRecordGuid(TABLES.Patient.TableName, "PatientMergeDate IS NULL");

			DataTable dtP = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			DataTable dtPT = SprocHelper.GetInsertPatientTreatmentSprocTable(patientTreatmentGuid, patientGuid);
			DataTable dtPO = SprocHelper.GetInsertPatientOrderSprocTable(patientOrderGuid, patientTreatmentGuid);
			DataTable dtOC = SprocHelper.GetHL7UpdateOrderedComponent(orderedComponentGuid, patientOrderGuid, false);
			DataTable dtOT = SprocHelper.GetHL7UpdateOrderedTest(orderedTestGuid, patientOrderGuid, true);

			dtOC.Rows[0][TABLES.OrderedComponent.SpecimenStatusCode] = "R";
			dtOC.Rows[0][TABLES.OrderedComponent.OrderStatusCode] = "P";
			dtOC.Rows[0][TABLES.OrderedComponent.PreOpIndicator] = false;
			dtOC.Rows[0][TABLES.OrderedComponent.RequiredDatetime] = DateTime.Now.AddDays(1);

			ArrayList dtArray = new ArrayList();
			ArrayList sprocArray = new ArrayList();

			if (newPatient)
			{
				dtArray.Add(dtP);
				sprocArray.Add(SPROCS.InsertPatient.StoredProcName);
			}

			dtArray.Add(dtPT);
			sprocArray.Add(SPROCS.InsertPatientTreatment.StoredProcName);
			dtArray.Add(dtPO);
			sprocArray.Add(SPROCS.InsertPatientOrder.StoredProcName);

			dtArray.Add(dtOC);
			sprocArray.Add(SPROCS.HL7UpdateOrderedComponent.StoredProcName);
			dtArray.Add(dtOT);
			sprocArray.Add(SPROCS.HL7UpdateOrderedTest.StoredProcName);

			int RC = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			Assert.AreEqual(0, RC, "DB Return Code");

		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="newPatient"></param>
		/// <returns>Guid [] (OrderedComponentGuid, OrderedTestGuid)</returns>
		public static Guid [] CreateNewOrder()
		{
			Guid orderedComponentGuid = Guid.NewGuid();
			Guid orderedTestGuid = Guid.NewGuid();
			Guid patientOrderGuid = Guid.NewGuid();
			
			Guid patientTreatmentGuid = Guid.NewGuid();
			Guid patientGuid = GetRandomRecordGuid(TABLES.Patient.TableName, "PatientMergeDate IS NULL");

			DataTable dtP = SprocHelper.GetInsertPatientSprocTable(patientGuid);
			DataTable dtPT = SprocHelper.GetInsertPatientTreatmentSprocTable(patientTreatmentGuid, patientGuid);
			DataTable dtPO = SprocHelper.GetInsertPatientOrderSprocTable(patientOrderGuid, patientTreatmentGuid);
			DataTable dtOC = SprocHelper.GetHL7UpdateOrderedComponent(orderedComponentGuid, patientOrderGuid, false);
			DataTable dtOT = SprocHelper.GetHL7UpdateOrderedTest(orderedTestGuid, patientOrderGuid, true);

			dtOC.Rows[0][TABLES.OrderedComponent.SpecimenStatusCode] = "R";
			dtOC.Rows[0][TABLES.OrderedComponent.OrderStatusCode] = "P";
			dtOC.Rows[0][TABLES.OrderedComponent.PreOpIndicator] = false;
			dtOC.Rows[0][TABLES.OrderedComponent.RequiredDatetime] = DateTime.Now.AddDays(1);

			ArrayList dtArray = new ArrayList();
			ArrayList sprocArray = new ArrayList();

			dtArray.Add(dtPT);
			sprocArray.Add(SPROCS.InsertPatientTreatment.StoredProcName);
			dtArray.Add(dtPO);
			sprocArray.Add(SPROCS.InsertPatientOrder.StoredProcName);

			dtArray.Add(dtOC);
			sprocArray.Add(SPROCS.HL7UpdateOrderedComponent.StoredProcName);
			dtArray.Add(dtOT);
			sprocArray.Add(SPROCS.HL7UpdateOrderedTest.StoredProcName);

			int RC = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			Assert.AreEqual(0, RC, "DB Return Code");

			return new Guid [] { orderedComponentGuid, orderedTestGuid };
		}

		/// <summary>
		/// Creates a patient Assignment for the BloodUnit
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		/// <returns></returns>
		public static void CreatePatientAssignment(Guid exitingbloodUnitGuid, Guid NEWpatientGuid, bool createCompatibleXM, bool createTAS)
		{
			Guid ptGuid = Guid.NewGuid();
			Guid poGuid = Guid.NewGuid();
			Guid ocGuid = Guid.NewGuid();
			Guid ouCGuid = Guid.NewGuid();
			Guid ouTGuid = Guid.NewGuid();
			Guid otGuid = Guid.NewGuid();
			Guid psGuid = Guid.NewGuid();

			string xmResult = (createCompatibleXM) ? "C" : "I";

			//Create a brand new patient to build off of so we don't mess with existing data
			DataTable dtPatient = SprocHelper.GetInsertPatientSprocTable(NEWpatientGuid);
			DataTable dtPatientTreatment = SprocHelper.GetInsertPatientTreatmentSprocTable(ptGuid, NEWpatientGuid);
			DataTable dtPatientOrder = SprocHelper.GetInsertPatientOrderSprocTable(poGuid, ptGuid);
			
			DataTable dtOrderedComponent = SprocHelper.GetHL7UpdateOrderedComponent(ocGuid, poGuid, false);
			DataTable dtPatientSpecimen = SprocHelper.GetInsertPatientSpecimenSprocTable(psGuid, NEWpatientGuid, GetNewSpecimenUid());
			DataTable dtOrderedTest	= SprocHelper.GetInsertOrderedTestSprocTable(otGuid, poGuid, psGuid); 
			DataTable dtOrderedUnitC = SprocHelper.GetInsertOrderedUnitSprocTable2(ouCGuid, ocGuid, exitingbloodUnitGuid, Guid.Empty, xmResult);
			DataTable dtOrderedUnitT = SprocHelper.GetInsertOrderedUnitSprocTable2(ouTGuid, Guid.Empty, exitingbloodUnitGuid, otGuid, xmResult);

			System.Collections.ArrayList dtArray = new ArrayList();
			System.Collections.ArrayList sprocArray = new ArrayList();

			dtArray.Add(dtPatient);
			sprocArray.Add(SPROCS.InsertPatient.StoredProcName);

			dtArray.Add(dtPatientTreatment);
			sprocArray.Add(SPROCS.InsertPatientTreatment.StoredProcName);

			dtArray.Add(dtPatientOrder);
			sprocArray.Add(SPROCS.InsertPatientOrder.StoredProcName);

			dtArray.Add(dtOrderedComponent);
			sprocArray.Add(SPROCS.HL7UpdateOrderedComponent.StoredProcName);

			dtArray.Add(dtPatientSpecimen);
			sprocArray.Add(SPROCS.InsertPatientSpecimen.StoredProcName);

			dtArray.Add(dtOrderedTest);
			sprocArray.Add(SPROCS.InsertOrderedTest.StoredProcName);

			dtArray.Add(dtOrderedUnitC);
			sprocArray.Add(SPROCS.InsertOrderedUnit.StoredProcName);

			dtArray.Add(dtOrderedUnitT);
			sprocArray.Add(SPROCS.InsertOrderedUnit.StoredProcName);

			//98-102 = XM results
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable2(Guid.NewGuid(), exitingbloodUnitGuid, 98, ((createCompatibleXM) ? "0" : "0"), ouCGuid, ocGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable2(Guid.NewGuid(), exitingbloodUnitGuid, 99, ((createCompatibleXM) ? "0" : "0"), ouCGuid, ocGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable2(Guid.NewGuid(), exitingbloodUnitGuid, 100, ((createCompatibleXM) ? "0" : "0"), ouCGuid, ocGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable2(Guid.NewGuid(), exitingbloodUnitGuid, 101, ((createCompatibleXM) ? "4" : "0"), ouCGuid, ocGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable2(Guid.NewGuid(), exitingbloodUnitGuid, 102, ((createCompatibleXM) ? "C" : "I"), ouCGuid, ocGuid));

			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable3(Guid.NewGuid(), exitingbloodUnitGuid, 164, ((createCompatibleXM) ? "0" : "0"), ouTGuid, otGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable3(Guid.NewGuid(), exitingbloodUnitGuid, 165, ((createCompatibleXM) ? "0" : "0"), ouTGuid, otGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable3(Guid.NewGuid(), exitingbloodUnitGuid, 166, ((createCompatibleXM) ? "0" : "0"), ouTGuid, otGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable3(Guid.NewGuid(), exitingbloodUnitGuid, 167, ((createCompatibleXM) ? "4" : "0"), ouTGuid, otGuid));
			dtArray.Add(SprocHelper.GetInsertBloodUnitTestSprocTable3(Guid.NewGuid(), exitingbloodUnitGuid, 168, ((createCompatibleXM) ? "C" : "I"), ouTGuid, otGuid));

			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);

			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);
			sprocArray.Add(SPROCS.InsertBloodUnitTest.StoredProcName);

			//TAS TESTS
			if (createTAS)
			{
				DataTable dtTAS1 = SprocHelper.GetInsertOrderedTestSprocTable(Guid.NewGuid(), poGuid, psGuid);
				if (!dtTAS1.Columns.Contains(TABLES.OrderedTest.OrderableTestId))
					dtTAS1.Columns.Add(TABLES.OrderedTest.OrderableTestId, typeof(int));

				dtTAS1.Rows[0][TABLES.OrderedTest.OrderableTestId] = 7;
				dtTAS1.Rows[0][TABLES.OrderedTest.OrderStatusCode] = "C";

				dtArray.Add(dtTAS1);
				sprocArray.Add(SPROCS.InsertOrderedTest.StoredProcName);
			}
			int retValue = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);

			Assert.AreEqual(0, retValue, "SprocHelper(" + exitingbloodUnitGuid.ToString() + ")");

			if (createTAS)
			{
				UnitTestUtil.RunSQL("UPDATE OrderedComponent SET SpecimenRequiredIndicator = 1 WHERE OrderedComponentGuid = '" + ocGuid.ToString() + "'");
				UnitTestUtil.RunSQL("UPDATE OrderedComponent SET PatientSpecimenGuid = '" + psGuid.ToString() + "' WHERE OrderedComponentGuid = '" + ocGuid.ToString() + "'");
			}
		}

		public static void CreateQCdRack()
		{
			Guid luckyRack = Guid.Empty;
			DataTable dtRacks = BOL.Rack.GetDailyQCRacks(DateTime.Now);
			foreach(DataRow drRack in dtRacks.Rows)
			{
				if (drRack.IsNull(TABLES.DailyQcWorklist.DailyUseIndicator) == false)
				{
					if( !(bool)drRack[TABLES.DailyQcWorklist.DailyUseIndicator] )
					{
						luckyRack = (Guid)drRack[TABLES.Rack.RackGuid];
						continue;
					}
				}
			}
			
			Guid rackGuid = DataCreator.GetRandomRecordGuid(TABLES.Rack.TableName, "RackName LIKE '[A-Z]'");
			if (luckyRack != Guid.Empty) rackGuid = luckyRack;

			BOL.Rack tmpRack = new BOL.Rack(rackGuid);

			Guid dailyControlGuid = Guid.NewGuid();
			BOL.DailyControl tmpDC = new BOL.DailyControl();
			DataTable existingDC = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM DailyControl WHERE RecordStatusCode = 'A' AND DivisionCode = '" + UnitTestConfig.TestDivisionCode + "'");
			if (existingDC.Rows.Count > 0)
			{
				tmpDC = new BOL.DailyControl(existingDC.Rows[0]);
			}
			else
			{
				tmpDC.DailyControlGuid = Guid.NewGuid();
				tmpDC.CommercialTemplateIndicator = true;
				tmpDC.DailyAlertTime = DateTime.Now; 
				tmpDC.NumberOfRacks = new Random().Next(1,24);

				DataTable dtEMedia = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM " + TABLES.EnhancementMedia.TableName);
				tmpDC.PrimaryEnhancementMedia = new BOL.EnhancementMedia(dtEMedia.Rows[0]);
				tmpDC.RackNamesIndicator = true;
				tmpDC.RecordStatusCode = COMMON.RecordStatusCode.Active;
			}
		
			tmpDC.PopulateRackData(tmpDC.DailyControlGuid);
			ArrayList alReagents = BOL.ReagentType.GetReagentQCData(true, UnitTestConfig.TestDivisionCode);
			tmpDC.SaveAllDailyControlData(alReagents, true, Common.UpdateFunction.UnitTests);

			//Create the DailyQCWorklist 
			BOL.Rack.UpdateRackDailyUse(Guid.Empty, true, true, Common.UpdateFunction.UnitTests, new byte[8], rackGuid, DateTime.Now.AddHours(1));

			//Create the Reagents -- TODO: get rid of hardcoding
			DataTable dtReagents = BOL.Reagent.InitializeReagentsAndManufacturersTable();
			GetNewReagentTableRow(dtReagents,48, "Anti-A");
			GetNewReagentTableRow(dtReagents,50, "Anti-B");
			GetNewReagentTableRow(dtReagents,52, "Anti-D");
			GetNewReagentTableRow(dtReagents,56, "CC");
			GetNewReagentTableRow(dtReagents,57, "LISS");
			GetNewReagentTableRow(dtReagents,59, "AHG, PS");
			GetNewReagentTableRow(dtReagents,60, "QC Kit");
			GetNewReagentTableRow(dtReagents,62, "Reverse ABO Typing Cells");
			GetNewReagentTableRow(dtReagents,63, "Screening Cells");

			//Insert all the Reagents
			if (!BOL.Reagent.InsertNewReagent(dtReagents, BOL.ReagentType.GetReagentTypes(), Common.UpdateFunction.UnitTests)) throw new Exception("InsertNewReagent failed");

			//Insert the Daily QC
			CreateDailyQC(rackGuid);
		}

		private static void CreateDailyQC(Guid rackGuid)
		{
			DataTable dtRackTestResults = new DataTable();
			dtRackTestResults.Columns.Add("Insert", typeof(bool));
			dtRackTestResults.Columns.Add("TestWithReagent", typeof(bool));
			dtRackTestResults.Columns.Add("TestWithReagentGuid", typeof(Guid));
			dtRackTestResults.Columns.Add(TABLES.RackLot.ReagentGuid, typeof(Guid));
			dtRackTestResults.Columns.Add(TABLES.ReagentType.ReagentTypeName);
			dtRackTestResults.Columns.Add(TABLES.RackLot.TestWithId);
			dtRackTestResults.Columns.Add(TABLES.RackLot.TestWithName);
			dtRackTestResults.Columns.Add(ARTIFICIAL.SetVialId);
			dtRackTestResults.Columns.Add(TABLES.RackLot.ExpectedMinimumReactions);
			dtRackTestResults.Columns.Add(TABLES.ReactivityPhase.ReactivityPhaseText);
			dtRackTestResults.Columns.Add(TABLES.ReactivityPhase.ReactivityPhaseCode);
			dtRackTestResults.Columns.Add(ARTIFICIAL.IsResults);
			dtRackTestResults.Columns.Add(ARTIFICIAL.IatResults);
			dtRackTestResults.Columns.Add(ARTIFICIAL.CcResults);
			dtRackTestResults.Columns.Add("TestResultIndicator");
			dtRackTestResults.Columns.Add(ARTIFICIAL.ISGuid);
			dtRackTestResults.Columns.Add(ARTIFICIAL.CCGuid);
			dtRackTestResults.Columns.Add("IatGuid");
			dtRackTestResults.Columns.Add(TABLES.ReagentType.ReagentTypeId);
			dtRackTestResults.Columns.Add("ReadOnlyIsResults", typeof(Boolean));
			dtRackTestResults.Columns.Add("ReadOnlyIatResults", typeof(Boolean));
			dtRackTestResults.Columns.Add("ReadOnlyCcResults", typeof(Boolean));
			dtRackTestResults.Columns.Add(ARTIFICIAL.IndicatorMessage);
			dtRackTestResults.Columns.Add(TABLES.RackLot.TestingMethodCode, typeof(string));

			DataTable dtReagents = BOL.Reagent.GetReagents(UnitTestConfig.TestDivisionCode);

			foreach(DataRow drReagentRecord in dtReagents.Rows)
			{
				bool isCC = false;
				bool isIAT = false;

				//Create a CC for 52 (Anti-D)
				isCC =  ((int)drReagentRecord[TABLES.ReagentType.ReagentTypeId] == 52);
				isIAT  =  ((int)drReagentRecord[TABLES.ReagentType.ReagentTypeId] == 63);

				
				if (isCC)
				{
					DataRow drCCRow = dtRackTestResults.NewRow();
					drCCRow[TABLES.RackLot.ReagentGuid] = drReagentRecord[TABLES.Reagent.ReagentGuid];
					drCCRow["Insert"] = true;
					drCCRow["TestWithReagent"] = false;
					drCCRow["ReagentTypeName"] = drReagentRecord[TABLES.ReagentType.ReagentTypeName];
					drCCRow[TABLES.RackLot.TestWithId] = 60;
					drCCRow[TABLES.RackLot.TestWithName] = drReagentRecord[TABLES.Reagent.ReagentName];
					drCCRow["SetVialId"] = "1";
					drCCRow["ExpectedMinimumReactions"] = "0";
					drCCRow["ReactivityPhaseText"] = "CC";
					drCCRow[TABLES.ReactivityPhase.ReactivityPhaseCode] = "1";
					drCCRow["IsResults"] = (isCC) ? "1" : "2";
					drCCRow["IatResults"] = (isCC) ? "1" : "2";
					drCCRow["CcResults"] = (isCC) ? "1" : "2";
					drCCRow["TestResultIndicator"] = "V";
					drCCRow[TABLES.ReagentType.ReagentTypeId] = drReagentRecord[TABLES.ReagentType.ReagentTypeId];
					drCCRow["ReadOnlyIsResults"] = true;
					drCCRow["ReadOnlyIatResults"] = true;
					drCCRow["ReadOnlyCcResults"] = false;
					drCCRow[ARTIFICIAL.IndicatorMessage] = "Configure CC";
					drCCRow[TABLES.RackLot.TestingMethodCode] = "T";
					dtRackTestResults.Rows.Add(drCCRow);
				}

				if (!isIAT)
				{
					DataRow drISRow = dtRackTestResults.NewRow();
					drISRow[TABLES.RackLot.ReagentGuid] = drReagentRecord[TABLES.Reagent.ReagentGuid];
					drISRow["Insert"] = true;
					drISRow["TestWithReagent"] = false;
					drISRow["ReagentTypeName"] = drReagentRecord[TABLES.ReagentType.ReagentTypeName];
					drISRow[TABLES.RackLot.TestWithId] = 60;
					drISRow[TABLES.RackLot.TestWithName] = drReagentRecord[TABLES.Reagent.ReagentName];
					drISRow["SetVialId"] = "1";
					drISRow["ExpectedMinimumReactions"] = (isCC) ? "0" : "2";
					drISRow["ReactivityPhaseText"] = "IS";
					drISRow[TABLES.ReactivityPhase.ReactivityPhaseCode] = "1";
					drISRow["IsResults"] = (isCC) ? "0" : "2";
					drISRow["IatResults"] = (isCC) ? "0" : "2";
					drISRow["CcResults"] = (isCC) ? "0" : "2";
					drISRow["TestResultIndicator"] = "V";
					drISRow[TABLES.ReagentType.ReagentTypeId] = drReagentRecord[TABLES.ReagentType.ReagentTypeId];
					drISRow["ReadOnlyIsResults"] = false;
					drISRow["ReadOnlyIatResults"] = true;
					drISRow["ReadOnlyCcResults"] = true;
					drISRow[ARTIFICIAL.IndicatorMessage] = "Configure IS";
					drISRow[TABLES.RackLot.TestingMethodCode] = "T";
					dtRackTestResults.Rows.Add(drISRow);
				}

				if (isIAT || isCC)
				{
					DataRow drIATRow = dtRackTestResults.NewRow();
					drIATRow[TABLES.RackLot.ReagentGuid] = drReagentRecord[TABLES.Reagent.ReagentGuid];
					drIATRow["Insert"] = true;
					drIATRow["TestWithReagent"] = false;
					drIATRow["ReagentTypeName"] = drReagentRecord[TABLES.ReagentType.ReagentTypeName];
					drIATRow[TABLES.RackLot.TestWithId] = 60;
					drIATRow[TABLES.RackLot.TestWithName] = drReagentRecord[TABLES.Reagent.ReagentName];
					drIATRow["SetVialId"] = "1";
					drIATRow["ExpectedMinimumReactions"] = (isCC) ? "0" : "2";
					drIATRow["ReactivityPhaseText"] = "IAT";
					drIATRow[TABLES.ReactivityPhase.ReactivityPhaseCode] = "1";
					drIATRow["IsResults"] = (isCC) ? "0" : "2";
					drIATRow["IatResults"] = (isCC) ? "0" : "2";
					drIATRow["CcResults"] = (isCC) ? "0" : "2";
					drIATRow["TestResultIndicator"] = "V";
					drIATRow[TABLES.ReagentType.ReagentTypeId] = drReagentRecord[TABLES.ReagentType.ReagentTypeId];
					drIATRow["ReadOnlyIsResults"] = true;
					drIATRow["ReadOnlyIatResults"] = false;
					drIATRow["ReadOnlyCcResults"] = true;
					drIATRow[ARTIFICIAL.IndicatorMessage] = "Configure IAT";
					drIATRow[TABLES.RackLot.TestingMethodCode] = "T";
					dtRackTestResults.Rows.Add(drIATRow);
				}

			}

			Common.WorkloadProcessID [] workloadProcessIds = new Common.WorkloadProcessID[1];
			DataCreator.CreateWorkloadProcess(Common.WorkloadProcessID.DailyRackQualityControlQC);
			workloadProcessIds[0] = Common.WorkloadProcessID.DailyRackQualityControlQC;

			DataTable dtRack = UnitTestUtil.RunSQLGetDataTable("SELECT TOP 1 * FROM RACK WHERE RackGuid = '" + rackGuid.ToString() + "'");

			bool bResult = BOL.RackLot.SaveRackTests(dtRackTestResults, rackGuid, (byte[])dtRack.Rows[0][TABLES.Rack.RowVersion], false, UnitTestConfig.TestDivisionCode, Common.UpdateFunction.UnitTests, workloadProcessIds, Environment.UserName , DateTime.Now, DateTime.Now.AddHours(5), Guid.Empty);
			if (!bResult) throw new Exception("CreateDailiyQC Failed!");
		}

		private static void GetNewReagentTableRow(DataTable dtReagents, int reagentTypeId, string reagentName)
		{
			DataRow drReagent = dtReagents.NewRow();
			drReagent[TABLES.Reagent.ReagentGuid] = Guid.NewGuid();
			drReagent[TABLES.ReagentType.ReagentTypeId] = reagentTypeId;
			drReagent[TABLES.Reagent.ReagentName] = reagentName;
			drReagent[TABLES.Reagent.ReceiptDate] = DateTime.Now;
			drReagent[TABLES.Reagent.ReceiptTechId] = Common.LogonUser.LogonUserName;
			drReagent[TABLES.Reagent.InvoiceNumber] = DateTime.Now.ToString();
			drReagent[TABLES.Reagent.LotNumber] = DateTime.Now.ToString();
			drReagent[TABLES.Reagent.LotExpirationDate] = DateTime.Now.AddYears(1);
			drReagent[TABLES.Reagent.NumberVialsReceived] = 5;
			drReagent[TABLES.Reagent.NumberVialsAvailable] = 5;
			drReagent[TABLES.Reagent.InspectionIndicator] = true;
			drReagent[TABLES.Reagent.ReagentManufacturerName] = "GetNewReagentTableRow";
			drReagent[TABLES.Reagent.Comments] = DateTime.Now.ToString();	//Required by sproc
			drReagent[TABLES.ReagentType.ReagentTypeName] = reagentName;
			drReagent[TABLES.VamcDivision.DivisionCode] = UnitTestConfig.TestDivisionCode;
			drReagent[TABLES.Reagent.RecordStatusCode] = "A";
			
			dtReagents.Rows.Add(drReagent);
		}

		public static void RefreshInterfaceControlTable()
		{
			UnitTests.UnitTestUtil.RunSQL("UPDATE InterfaceControl SET InterfaceFacilityId = '589', InterfaceIpAddress = '10.3.29.203', VbecsFacilityId = 'VBECS', InterfaceAdministratorName = 'Greg.Lohse@domain', InterfaceActiveIndicator = 1, LastUpdateDate = getutcdate(), VbecsIpAddress='10.3.8.185', VbecsPortNumber='21995'");
			UnitTests.UnitTestUtil.RunSQL("UPDATE InterfaceControl SET InterfacePortNumber = 11993, InterfaceIpAddress = '10.3.29.203', LastUpdateDate = getutcdate() WHERE InterfaceControlId = 1");
			UnitTests.UnitTestUtil.RunSQL("UPDATE InterfaceControl SET InterfacePortNumber = 19998, InterfaceIpAddress = '10.3.21.93', LastUpdateDate = getutcdate() WHERE InterfaceControlId = 4");

		}

		public static void ResetVistaLinkControlTable()
		{
			UnitTests.UnitTestUtil.RunSQL("DELETE FROM VistaLinkControl");

			DAL.VistALink.OpenLibrary.ServerConnectionInfo _sciFromAppConfig = DAL.VistALink.OpenLibrary.GlobalConfig.GetServerConnectionInfoByHandle( "Primary" );

			StringBuilder _sql = new StringBuilder();
			_sql.AppendFormat( "insert into {0}( ", TABLES.VistaLinkControl.TableName );
			_sql.AppendFormat( "{0},", TABLES.VistaLinkControl.ServerAppHandle );
			_sql.AppendFormat( "{0},", TABLES.VistaLinkControl.ServerAlias );
			_sql.AppendFormat( "{0},", TABLES.VistaLinkControl.VistaIpAddress );
			//_sql.AppendFormat( "{0},", TABLES.VistaLinkControl.VistaDomainName );
			_sql.AppendFormat( "{0},", TABLES.VistaLinkControl.VistaPortNumber );
            _sql.AppendFormat("{0},", TABLES.VistaLinkControl.VbecsServiceIpAddress);   //new 2.0.0.4
            _sql.AppendFormat("{0},", TABLES.VistaLinkControl.VbecsServicePortNumber);  //new 2.0.0.4

			_sql.AppendFormat( "{0},", TABLES.VistaLinkControl.LastUpdateUser );
			_sql.AppendFormat( "{0},", TABLES.VistaLinkControl.LastUpdateDate );
			_sql.AppendFormat( "{0}", TABLES.VistaLinkControl.LastUpdateFunctionId );
			_sql.Append( ") values ( " );
			_sql.AppendFormat( "'{0}',", _sciFromAppConfig.Handle );
			_sql.AppendFormat( "'{0}',", _sciFromAppConfig.Alias );
			_sql.AppendFormat( "'{0}',", _sciFromAppConfig.IPAddress.ToString() );
			//_sql.AppendFormat( "'{0}',", _sciFromAppConfig.Domain.ToString() );
			_sql.AppendFormat( "{0},",_sciFromAppConfig.PortNumber.ToString() );
            _sql.AppendFormat("'{0}',", "10.3.9.185");     //new 2.0.0.4
            _sql.AppendFormat("'{0}',", "11995");         //new 2.0.0.4
			_sql.AppendFormat( "'{0}',", "UnitTestRestore" );
			_sql.Append( "getdate()," );
			_sql.AppendFormat( "{0}", 0 );
			_sql.Append( ")" );

			UnitTests.UnitTestUtil.RunSQL(_sql.ToString());
		}


	}
}
#endif

			